package com.openexchange.office.rt2.hazelcast;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.osgi.framework.BundleException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.hazelcast.core.IMap;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeDocsPredicate;
import com.openexchange.office.tools.annotation.RegisteredService;

@Service
@RegisteredService(registeredClass=RT2DocOnNodeMap.class)
public class HzDocOnNodeMap extends RT2HazelcastService implements RT2DocOnNodeMap
{
	private static final Logger log = LoggerFactory.getLogger(HzDocOnNodeMap.class);

    @Override
	public void afterPropertiesSet() throws Exception {
        mapIdentifier = HzHelper.discoverMapName(hzInstance.getConfig(), "rt2DocOnNodeMap-");
        if(StringUtils.isEmpty(mapIdentifier)) {
            final String msg = "Distributed rt2 doc node map couldn't be found in hazelcast configuration";
            throw new IllegalStateException(msg, new BundleException(msg, BundleException.ACTIVATOR_ERROR));
        }
        log.info("Registered rt2DocOnNodeMap to Hazelcast");
	}

	//-------------------------------------------------------------------------
	@Override
	public String get(String sDocUID) {
        final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
        return allDocToNodeMappings.get(sDocUID);
	}

    //-------------------------------------------------------------------------
	@Override
	public Map<String, String> get(Collection<String> aDocUIDs)  {
              Map<String, String> aFoundNodesMap = new HashMap<>();
        final Set<String>         aDocUIDSet     = new HashSet<>();

        for (final String sDocUID : aDocUIDs) {
        	aDocUIDSet.add(sDocUID);
        }

        if (!aDocUIDSet.isEmpty()) {
            final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
            aFoundNodesMap = allDocToNodeMappings.getAll(aDocUIDSet);
        }

        return aFoundNodesMap;
	}

    //-------------------------------------------------------------------------
	@Override
	public String set(String sDocUID)  {
	    String thisNodeUUID = hzInstance.getCluster().getLocalMember().getUuid();
        final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
        String sPrevNodeUID = allDocToNodeMappings.get(sDocUID);

        if (sPrevNodeUID != null) {
            // but the previous resource provides a presence
        	allDocToNodeMappings.put(sDocUID, thisNodeUUID);
        } else {
        	sPrevNodeUID = allDocToNodeMappings.putIfAbsent(sDocUID, thisNodeUUID);
        }
        return sPrevNodeUID;
	}

    //-------------------------------------------------------------------------
    @Override
    public String set(String docUID, String nodeUUID) {
        final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
        String sPrevNodeUID = allDocToNodeMappings.get(docUID);
        if(sPrevNodeUID != null) {
            return allDocToNodeMappings.put(docUID, nodeUUID);
        }
        return null;
    }

    //-------------------------------------------------------------------------
	@Override
	public void setIfAbsent(String sDocUID) {
	   final String thisNodeUUID = hzInstance.getCluster().getLocalMember().getUuid();
       final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
       String aPrevNodeUUID = allDocToNodeMappings.get(sDocUID);

       if (aPrevNodeUUID == null) {
          	allDocToNodeMappings.put(sDocUID, thisNodeUUID);
        }
	}

    //-------------------------------------------------------------------------
	@Override
	public String remove(String sDocUID) {
        final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
        String sPrevNodeUUID = allDocToNodeMappings.get(sDocUID);

        if (sPrevNodeUUID != null) {
           	allDocToNodeMappings.remove(sDocUID);
        }
        return sPrevNodeUUID;
	}

    //-------------------------------------------------------------------------
	@Override
	public Map<String, String> remove(Collection<String> aDocUIDs)  {
        final Map<String, String> aRemovedMap = new HashMap<>();
        final Set<String>         aDocUIDSet  = new HashSet<>();

        for (final String sDocUID : aDocUIDs) {
        	aDocUIDSet.add(sDocUID);
        }

        if (!aDocUIDSet.isEmpty()) {
            final IMap<String, String> allDocToNodeMappings = getDocNodeMapping();
            final Map<String, String>  aMatchStates         = allDocToNodeMappings.getAll(aDocUIDSet);

            if (aMatchStates != null) {
                for (Entry<String, String> entry : aMatchStates.entrySet()) {
                	final String sFoundUID = entry.getKey();
                	allDocToNodeMappings.remove(sFoundUID);
                	aRemovedMap.put(sFoundUID, entry.getValue());
                }
            }
        }

        return aRemovedMap;
	}

    //-------------------------------------------------------------------------
    @Override
	public Set<String> getDocsOfMember() {
        final String thisNodeUUID = hzInstance.getCluster().getLocalMember().getUuid();
    	return getDocsOfMember(thisNodeUUID);
    }

    //-------------------------------------------------------------------------
    @Override
	public Set<String> getDocsOfMember(final String sNodeUUID)
    {
        final IMap<String, String> allResources = getDocNodeMapping();
        final PortableNodeDocsPredicate nodeDocsPredicate = new PortableNodeDocsPredicate(sNodeUUID);
        final Set<Entry<String, String>> matchingResources = allResources.entrySet(nodeDocsPredicate);

        final Set<String> foundIds = new HashSet<String>();
        final Iterator<Entry<String, String>> iterator = matchingResources.iterator();
        while(iterator.hasNext()) {
            try {
                Entry<String, String> next = iterator.next();

                foundIds.add(next.getKey());
            } catch (Exception e) {
            	log.error("Couldn't add resource that was found for member " + sNodeUUID, e);
            }
        }

        return foundIds;
    }

    //-------------------------------------------------------------------------
    @Override
    public String getUniqueMapName() {
        return mapIdentifier;
	}

    //-------------------------------------------------------------------------
    public IMap<String, String> getDocNodeMapping() {
        return hzInstance.getMap(mapIdentifier);
    }

  //-------------------------------------------------------------------------
	@Override
	public Set<String> getMember() {
		return new HashSet<>(getDocNodeMapping().values());
	}

}
