/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.core.ws;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.glassfish.grizzly.websockets.DataFrame;
import org.glassfish.grizzly.websockets.WebSocket;
import org.glassfish.grizzly.websockets.WebSocketListener;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.openexchange.log.LogProperties;
import com.openexchange.office.document.api.AdvisoryLockMode;
import com.openexchange.office.rt2.core.cache.RT2HazelcastHelperService;
import com.openexchange.office.rt2.core.exception.RT2SessionInvalidException;
import com.openexchange.office.rt2.core.exception.RT2TypedException;
import com.openexchange.office.rt2.core.metric.DocProxyRequestMetricService;
import com.openexchange.office.rt2.core.metric.DocProxyResponseMetricService;
import com.openexchange.office.rt2.core.metric.WebsocketResponseMetricService;
import com.openexchange.office.rt2.core.proxy.RT2DocProxy;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyLogInfo.Direction;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyRegistry;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.annotation.ShutdownOrder;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.log.LogMethodCallHelper;
import com.openexchange.office.tools.common.string.StringHelper;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;

@Service
@ShutdownOrder(value=-3)
@RegisteredService
public class RT2WebSocketListener implements WebSocketListener, DisposableBean {

	private static final Logger log = LoggerFactory.getLogger(RT2WebSocketListener.class);
	private static final byte[] data = new byte[] { 0 };

	private final ExecutorService onWebSocketMsgThreadPool = Executors.newFixedThreadPool(10, new ThreadFactoryBuilder("RT2WebSocketListenerOnMsg-%d").build());

	//--------------------------Services--------------------------------------
	@Autowired
	private RT2SessionValidator sessionValidator;

	@Autowired
	private RT2SessionCountValidator sessionCountValidator;

	@Autowired
	private RT2WSChannelDisposer channelDisposer;

	@Autowired
	private RT2DocProxyRegistry docProxyRegistry;

	@Autowired
    private WebsocketResponseMetricService websocketResponseMetriceService;

	@Autowired
    private DocProxyRequestMetricService docProxyRequestMetricService;

	@Autowired
    private DocProxyResponseMetricService docProxyResponseMetricService;

	@Autowired
	private RT2WSApp rt2WsApp;

	@Autowired
	private RT2HazelcastHelperService rt2HazelcastHelperService;

	//------------------------------------------------------------------------

	@Override
	public void destroy() throws Exception {
		onWebSocketMsgThreadPool.shutdown();
	}

	@Override
	public void onClose(WebSocket socket, DataFrame frame) {
		final RT2EnhDefaultWebSocket rt2WebSocket = (RT2EnhDefaultWebSocket) socket;
		// ATTENTION: We should only trigger an offline if we don't have a websocket
		// instance in our websocket container. The Grizzly uses for a certain time
		// span more than one  websocket for the same channel. Therefore we need to
		// use the following check.
		if (!rt2WsApp.contains(rt2WebSocket.getId())) {
			final Set<Pair<RT2CliendUidType, RT2DocUidType>> assocClients = channelDisposer.getClientDocPairsAssociatedToChannel(rt2WebSocket.getId());
			assocClients.forEach(p -> {
				log.debug("RT2: ... client '{}' switch into offline mode for doc '{}'", p.getKey(), p.getValue());
				RT2DocProxy docProxy = null;
				try {
					docProxy = docProxyRegistry.getDocProxy(p.getKey(), p.getValue());
					if (docProxy != null) {
						docProxy.goOffline();
					}
				} catch (Throwable ex) {
	                com.openexchange.exception.ExceptionUtils.handleThrowable(ex);
	                log.error("RT2: ... client '"+p.getKey()+"' failed on switching into offline mode for doc '"+p.getValue() + ", msgs: " + (docProxy == null ? "" : docProxy.formatMsgsLogInfo()), ex);
	            }
			});
		}
		this.channelDisposer.removeChannel(rt2WebSocket);
		sessionCountValidator.removeSession(rt2WebSocket.getId());
	}

	@Override
	public void onConnect(WebSocket socket) {
		// Nothing todo
	}

	@Override
	public void onMessage(WebSocket socket, String msg) {
		onWebSocketMsgThreadPool.execute(() -> {
			onMessageSync(socket, msg);
		});
	}

	public void onMessageSync(WebSocket socket, String msg) {
		RT2EnhDefaultWebSocket rt2WebSocket = (RT2EnhDefaultWebSocket) socket;
		rt2WebSocket.setCurrentlyProcessingRequest(true);
		rt2WebSocket.updateLastRecMsgTime();
		try {
			final RT2Message request = RT2MessageFactory.fromJSONString(msg);
			if (request.getType() == RT2MessageType.PONG) {
				log.debug("RT2: pong msg {} received from ws {}", msg, socket);
				onPong(socket, data);
				Set<Pair<RT2CliendUidType, RT2DocUidType>> clientDocPairsAssociatedToChannel = channelDisposer.getClientDocPairsAssociatedToChannel(rt2WebSocket.getId());
				clientDocPairsAssociatedToChannel.stream().forEach(p -> {
					RT2DocProxy docProxy =  docProxyRegistry.getDocProxy(p.getKey(), p.getValue());
					if (docProxy != null) {
						docProxy.goOnline();
					}
				});
			} else {
				setMDCVariables(request);
				startMetricsMeasurement(request);
				impl_handleRequest(socket, request);
			}

			if (!rt2WebSocket.isAvaible()) {
				rt2WebSocket.setAvaible(true);
				channelDisposer.notifyUnavailability(Direction.FROM_WS, rt2WebSocket, 0);
			}
        } catch (final Throwable ex) {
            log.error(ex.getMessage(), ex);
            // no further handling !
            // All responses was send within impl_handleRequest() already !
        } finally {
            LogProperties.removeLogProperties();
            rt2WebSocket.setCurrentlyProcessingRequest(false);
        }
	}

	@Override
	public void onMessage(WebSocket socket, byte[] bytes) {
		onMessage(socket, new String(bytes));
	}

	@Override
	public void onPong(WebSocket socket, byte[] bytes) {
		RT2EnhDefaultWebSocket rt2WebSocket = (RT2EnhDefaultWebSocket) socket;
		rt2WebSocket.updateLastRecMsgTime();
		log.debug("RT2: websocket channel got pong response [{}]", StringHelper.toHexString(bytes, true));
		if (!rt2WebSocket.isAvaible()) {
			rt2WebSocket.setAvaible(true);
			log.info("Notifiy unavilability for websocket with id {} because of onPong", rt2WebSocket.getId());
			channelDisposer.notifyUnavailability(Direction.INTERNAL, rt2WebSocket, 0);
		}
	}

	public void emergencyLeave(WebSocket socket, String docUID, String clientUID, String sessionId, JSONObject leaveData) throws Exception {
		  // ATTENTION: This is a special method called out-side the web
		  // socket communication env to enable clients to send a last-message
		  // that notifies the RT2WSChannel that a client has to leave in an
		  // emergency case and won't be able to send/receive any more messages.
		  //
		  // The implementation needs to create a priority RT2Message and
		  // send it to the DocProcessor ignoring the current state machine
		  // state. It's also responsible to clean-up the resources that the
		  // client references when the response of the DocProcessor has been
		  // received.
		  final RT2Message aEmergencyLeaveRequest = RT2MessageFactory.newMessage(RT2MessageType.REQUEST_EMERGENCY_LEAVE, new RT2CliendUidType(clientUID), new RT2DocUidType(docUID));
		  RT2MessageGetSet.setSessionID(aEmergencyLeaveRequest, sessionId);
		  aEmergencyLeaveRequest.setBody(leaveData);

		  impl_handleRequest(socket, aEmergencyLeaveRequest);
		}


    //-------------------------------------------------------------------------
    /** handle incoming client requests.
     *
     *  @param	sRequest [IN]
     *      the request from client side formated as JSON.
     */

	private void impl_handleRequest(WebSocket socket, final RT2Message request) {
		boolean failed = false;
		RT2DocProxy docProxy = null;
		RT2EnhDefaultWebSocket enhSocket = (RT2EnhDefaultWebSocket) socket;
        try {
        	channelDisposer.addDocUidToChannel(enhSocket, request.getClientUID(), request.getDocUID());
        	sessionValidator.validateSession(request, enhSocket);
        	sessionCountValidator.addSession(request.getSessionID(), enhSocket.getId());
        	if (RT2MessageType.REQUEST_JOIN.equals(request.getType())) {
        		channelDisposer.addDocUidToChannel(enhSocket, request.getClientUID(), request.getDocUID());
        		docProxy = docProxyRegistry.getOrCreateDocProxy(request, true, enhSocket.getId(), request.getSessionID());
        	} else {
        		docProxy = docProxyRegistry.getDocProxy(RT2DocProxy.calcID(request.getClientUID(), request.getDocUID()));
        	}
        	if (docProxy == null) {
        		if (!RT2MessageType.ACK_SIMPLE.equals(request.getType()) && !RT2MessageType.NACK_SIMPLE.equals(request.getType())) {
        			throw new RT2TypedException(ErrorCode.GENERAL_CLIENT_UID_UNKNOWN_ERROR, new ArrayList<>());
        		}
        		return;
        	}
        	docProxy.addMsgFromWebSocket(request);
            switch (request.getType()) {
                case REQUEST_JOIN:
                    if (enhSocket.getAdvisoryLockMode() == AdvisoryLockMode.ADVISORY_LOCK_MODE_ORIGIN)
                        impl_extendRequestWithHostName(enhSocket, request);
                	docProxy.join(request);
                    break;
                case REQUEST_OPEN_DOC:
                    if (enhSocket.getAdvisoryLockMode() == AdvisoryLockMode.ADVISORY_LOCK_MODE_ORIGIN)
                        impl_extendRequestWithHostName(enhSocket, request);
                    docProxy.open(request);
                    break;
                case REQUEST_CLOSE_DOC:
                	docProxy.close(request);
                    break;
                case REQUEST_LEAVE:
                case REQUEST_EMERGENCY_LEAVE:
                	docProxy.leave(request);
                    break;
                case REQUEST_ABORT_OPEN: docProxy.abortOpen(request);
                    break;
                default:                 docProxy.sendRequest(request);
                    break;
            }
            websocketResponseMetriceService.incMessageSucc(request.getMessageID());
        }
        catch (RT2SessionInvalidException ex) {
        	failed = true;
        	if (ex.isLog()) {
        		log.info("Session invalid exception " + ex.getMessage() + " occured during processing request " + request.toString());
        	}
        	impl_handleRequestErrors (request, ex, enhSocket);
        }
        catch (final Throwable ex) {
        	failed = true;
            // some rt2 typed exception are just informational - check ex type
            if (RT2TypedException.class.isAssignableFrom(ex.getClass ())) {
                final RT2TypedException aRT2Ex = (RT2TypedException) ex;
                final ErrorCode aError = aRT2Ex.getError();

                if (aError.isWarning())
                    log.debug(ex.getMessage() + "; request: " + request.getHeader() + ", msgs: " + (docProxy == null ? "" : docProxy.formatMsgsLogInfo()));
                else
                    log.error("Exception " + ex.getMessage() + " occured during processing request " + request.getHeader() + ", msgs: " + (docProxy == null ? "" : docProxy.formatMsgsLogInfo()), ex);
            } else {
                log.error("Exception " + ex.getMessage() + " occured during processing request " + request.getHeader() + ", msgs: " + (docProxy == null ? "" : docProxy.formatMsgsLogInfo()), ex);
            }
            impl_handleRequestErrors (request, ex, enhSocket);
        } finally {
			if (failed) {
				websocketResponseMetriceService.incMessageFailed(request.getMessageID());
			}
			websocketResponseMetriceService.stopTimer(request.getMessageID());
		}
    }

	//-------------------------------------------------------------------------
	private void impl_extendRequestWithHostName(RT2EnhDefaultWebSocket enhSocket, RT2Message request) {
	    final String origin = enhSocket.getUpgradeRequest().getHeader("origin");
	    final String hostNameFromOrigin = extractHostFromOrigin(origin);
	    RT2MessageGetSet.setOriginAsHostName(request, hostNameFromOrigin);
	    log.debug("Enhanced request from client-uid {} with origin as hostname {}", request.getClientUID(), hostNameFromOrigin);
	}

    //-------------------------------------------------------------------------
    private void impl_handleRequestErrors (final RT2Message aRequest, final Throwable  ex, RT2EnhDefaultWebSocket webSocket) {
        final RT2Message aResponse = RT2MessageFactory.createResponseFromMessage(aRequest, RT2MessageType.RESPONSE_GENERIC_ERROR);
        ErrorCode aError = ErrorCode.GENERAL_UNKNOWN_ERROR;

        if (RT2TypedException.class.isAssignableFrom(ex.getClass ())) {
            final RT2TypedException aRT2Ex = (RT2TypedException) ex;
            aError = aRT2Ex.getError();

            RT2MessageGetSet.setError(aResponse, aError);

            if (aError.equals(ErrorCode.GENERAL_SESSION_INVALID_ERROR)) {
            	impl_sendSyncResponse(aResponse, webSocket);
                impl_autoCloseOneClient (aRequest.getClientUID(), aRequest.getDocUID());
            } else {
            	impl_sendResponse (aResponse);
            }
        } else {
            RT2MessageGetSet.setError(aResponse, aError);
            impl_sendResponse (aResponse);
        }
    }

    //-------------------------------------------------------------------------
    private  void impl_sendResponse(final RT2Message aResponse)
    {
        // Warum? Klingt auch so, dass alle HeaderInfo geloescht werden, was nicht der Fall ist
        RT2Message.cleanHeaderBeforeSendToClient(aResponse);

        docProxyResponseMetricService.stopTimer(aResponse.getMessageID());
        log.debug("RT2WSChannel: Sending message to ws-client: {}", aResponse.getHeader());
        channelDisposer.addMessageToResponseQueue(aResponse);
    }

    //-------------------------------------------------------------------------
    private void impl_sendSyncResponse(final RT2Message response, RT2EnhDefaultWebSocket webSocket)
    {
        // Warum? Klingt auch so, dass alle HeaderInfo geloescht werden, was nicht der Fall ist
        RT2Message.cleanHeaderBeforeSendToClient(response);

        docProxyResponseMetricService.stopTimer(response.getMessageID());
        try {
			Thread.sleep(500);
		} catch (InterruptedException e) {
		}
        log.debug("RT2WSChannel: Sending message sync to ws-client: {}", response.getHeader());
   		webSocket.send(response);
    }

    //-------------------------------------------------------------------------
    private  void impl_autoCloseOneClient (final RT2CliendUidType sClientUID, final RT2DocUidType docUid) {
        LogMethodCallHelper.logMethodCall(log, this, "impl_autoCloseOneClient", sClientUID, docUid);

        try {
            if (docUid != null) {
                log.debug("RT2: impl_autoCloseOneClient with client={} and doc={}", sClientUID, docUid);

                final RT2DocProxy aDocProxy = docProxyRegistry.getDocProxy(sClientUID, docUid);
                if (aDocProxy == null)
                    return;

                aDocProxy.closeHard(false);
                // TODO: do we really need to call this? should be done by the async file leave by the proxy itself?
                docProxyRegistry.deregisterDocProxy(aDocProxy, false);
            } else {
                log.debug("RT2: impl_autoCloseOneClient client={} has no bound doc - nothing to do", sClientUID);
            }
        }
        catch (IllegalArgumentException ex) {
            log.debug("RT2: impl_autoCloseOneClient - no client info found auto close already completed");
        }
        catch (Throwable ex) {
            com.openexchange.exception.ExceptionUtils.handleThrowable(ex);
            log.error("RT2: impl_autoCloseOneClient - releasing connection between client="+sClientUID+" and doc="+docUid+" was not successful due to exception", ex);
        }
    }

    //-------------------------------------------------------------------------
	private void setMDCVariables(final RT2Message aRequest) {
        if (aRequest.getDocUID() != null) {
            LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, aRequest.getDocUID().getValue());
        }
        if (aRequest.getClientUID() != null) {
            LogProperties.putProperty(LogProperties.Name.RT2_CLIENT_UID, aRequest.getClientUID().getValue());
        }
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "proxy_forward");
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, rt2HazelcastHelperService.getHazelcastLocalNodeUuid());
        LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, aRequest.getType().getValue());
    }

    //-------------------------------------------------------------------------
    private void startMetricsMeasurement(final RT2Message aRequest) {
    	if (websocketResponseMetriceService != null) {
    		websocketResponseMetriceService.startTimer(aRequest.getMessageID());
    	}
    	if (websocketResponseMetriceService != null) {
    		websocketResponseMetriceService.incMessagesReceived(aRequest.getMessageID());
    	}
    	if (docProxyRequestMetricService != null) {
    		docProxyRequestMetricService.startTimer(aRequest.getMessageID());
    	}
    }

	@Override
	public void onFragment(WebSocket socket, String fragment, boolean last) {
		// Nothing to do, implemented in RT2EnhDefaultWebSocket class
	}

	@Override
	public void onFragment(WebSocket socket, byte[] fragment, boolean last) {
		// Nothing to do, implemented in RT2EnhDefaultWebSocket class
	}

	@Override
	public void onPing(WebSocket socket, byte[] bytes) {
		// Nothing to do
	}

    //-------------------------------------------------------------------------
    private String extractHostFromOrigin(String origin) {
        String hostName = null;
        if (StringUtils.isNotEmpty(origin)) {
            try {
                hostName = new URL(origin).getHost();
            } catch (MalformedURLException e) {
                log.info("Exception caught trying to determine hostname from origin of upgrade-request", e);
            }
        } else {
            // fallback for not set origin to empty string
            hostName = "";
        }

        return hostName;
    }
}
