/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.ws;

import java.lang.ref.WeakReference;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

//=============================================================================
/**
 * A thread-safe associative storage for RT2EnhDefaultWebSocket instances. It provides
 * methods to search, put or remove instances and functions to retrieve collections
 * of web sockets filtered by certain arguments.
 * 
 * @author Carsten Driesner 
 *
 */
public class ChannelStore {
    private static final Logger log = LoggerFactory.getLogger(ChannelStore.class);    

    private static class ChannelData {
    	public WeakReference<RT2EnhDefaultWebSocket> ws;
    	public long creationTime;

    	public ChannelData(RT2EnhDefaultWebSocket ws) {
    		this.ws = new WeakReference<>(ws);
    		this.creationTime = ws.getCreationTime();
    	}
    };

    private final Map<RT2ChannelId, ChannelData> channels = new HashMap<>();

    public ChannelStore() {
    	// nothing to do
    }

    public int size() {
    	synchronized(channels) {
        	return channels.size();
    	}
    }

    public boolean containsChannel(RT2ChannelId channelId) {
    	synchronized(channels) {
    		return channels.containsKey(channelId);
    	}
    }

    public WeakReference<RT2EnhDefaultWebSocket> get(RT2ChannelId channelId) {
    	synchronized(channels) {
        	final ChannelData channelData = channels.get(channelId);
        	if (channelData != null)
        		return channelData.ws;
    	}

    	return null;
    }

    public WeakReference<RT2EnhDefaultWebSocket> put(RT2ChannelId channelId, RT2EnhDefaultWebSocket ws) {
    	synchronized(channels) {
    		final ChannelData oldData = channels.put(channelId, new ChannelData(ws));
    		if (oldData != null) {
    			return oldData.ws;
    		}
    	}

		return null;
    }

    public boolean remove(RT2EnhDefaultWebSocket ws) {
    	if (ws != null) {
    		synchronized(channels) {
        		final ChannelData channelData = channels.get(ws.getId());
        		if ((channelData != null) && (channelData.creationTime == ws.getCreationTime())) {
        			channels.remove(ws.getId());
        			log.debug("websocket {} with id {} removed from ChannelStore");
        			return true;
        		}
    		}
    	}

    	return false;
    }

    public Collection<WeakReference<RT2EnhDefaultWebSocket>> values() {
    	synchronized(channels) {
    		Collection<ChannelData> coll = channels.values();
    		return coll.stream()
    				   .map(d -> d.ws)
    				   .collect(Collectors.toList());
    	}
    }

	public List<RT2ChannelId> getAllWSChannelIds() {
		synchronized(channels) {
			return channels.entrySet().stream().filter(e -> {
				return e.getValue().ws.get() != null;
			}).map(e -> e.getKey()).collect(Collectors.toList());
		}
    }

    public Set<RT2ChannelId> getActiveWebsocketIds() {
    	Set<RT2ChannelId> res;

    	synchronized(channels) {
            res = channels.entrySet().stream().filter(e -> {
    		    RT2EnhDefaultWebSocket ws = e.getValue().ws.get();
        		if (ws != null) {
        			final boolean available = ws.isAvaible();
        			log.trace("websocket {} with id {} found in ChannelStore, available state {}", ws, e.getKey().toString(), available);
        			return available;
        		} else {
        			log.debug("websocket {} with id {} found in ChannelStore, but instance null", ws, e.getKey());
        		}
        		return false;
        	}).map(e -> e.getKey())
        	  .collect(Collectors.toSet());
    	}

    	return res;
    }

}
