/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.proxy;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;

import org.apache.commons.lang3.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.jms.listener.DefaultMessageListenerContainer;
import org.springframework.stereotype.Service;
import org.springframework.util.backoff.ExponentialBackOff;

import com.openexchange.log.LogProperties;
import com.openexchange.logging.MDCEnabledThreadGroup;
import com.openexchange.office.rt2.core.cache.RT2HazelcastHelperService;
import com.openexchange.office.rt2.core.jms.RT2JmsDestination;
import com.openexchange.office.rt2.core.metric.DocProxyResponseMetricService;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.BroadcastMessage;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.BroadcastMessageReceiver;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.MessageType;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessagePostProcessor;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2ErrorCodeType;
import com.openexchange.office.rt2.protocol.value.RT2MessageIdType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.protocol.value.RT2SeqNumberType;
import com.openexchange.office.tools.annotation.ShutdownOrder;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.jms.JmsMessageListener;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;
import com.openexchange.office.tools.jms.PooledConnectionFactoryProxy;

@Service
@ShutdownOrder(value=-5)
public class RT2DocProxyJmsConsumer implements MessageListener, JmsMessageListener, InitializingBean, DisposableBean {

    private static final Logger log = LoggerFactory.getLogger(RT2DocProxyJmsConsumer.class);
   
    private String nodeUUID;
    
	//--------------------------Services--------------------------------------
    @Autowired
    private RT2DocProxyRegistry rt2DocRegistry;

    @Autowired
    private DocProxyResponseMetricService docProxyResponseMetricService;
    
    @Autowired
    private PooledConnectionFactoryProxy pooledConnectionFactoryProxy;    
    
    @Autowired
    private RT2HazelcastHelperService hazelcastHelperService;
    
	//------------------------------------------------------------------------    
    
    private Map<String, Long> receivedMessages = new HashMap<>();

    private Thread cleanupThread;

    private DefaultMessageListenerContainer msgListenerCont;

	@Override
	public void afterPropertiesSet() throws Exception {
		this.nodeUUID = hazelcastHelperService.getHazelcastLocalNodeUuid();
		
	}

	//-------------------------------------------------------------------------    
	@Override
	public void startReceiveMessages() {
        if (msgListenerCont == null) {
            msgListenerCont = new DefaultMessageListenerContainer();
            ExponentialBackOff exponentialBackOff = new ExponentialBackOff();
            exponentialBackOff.setMaxInterval(60000);
            msgListenerCont.setBackOff(exponentialBackOff);            
            msgListenerCont.setConnectionFactory(pooledConnectionFactoryProxy.getPooledConnectionFactory());
            msgListenerCont.setConcurrentConsumers(3);
            msgListenerCont.setDestination(RT2JmsDestination.clientResponseTopic);
            msgListenerCont.setMaxConcurrentConsumers(3);
            msgListenerCont.setPubSubDomain(true);
            msgListenerCont.setAutoStartup(true);
            msgListenerCont.setupMessageListener(this);
            msgListenerCont.setTaskExecutor(new SimpleAsyncTaskExecutor(new ThreadFactoryBuilder("RT2DocProcessorJmsConsumer-%d").build()));
            msgListenerCont.afterPropertiesSet();            
            msgListenerCont.start();

            cleanupThread = new Thread(new MDCEnabledThreadGroup(), new CleanupThread());
            cleanupThread.setName(cleanupThread.getClass().getName());
            cleanupThread.setDaemon(true);
            cleanupThread.start();
        }
	}
    
	//-------------------------------------------------------------------------    
    @Override
	public void destroy() throws Exception {
        if (msgListenerCont != null) {
            msgListenerCont.destroy();
            msgListenerCont = null;
        }
        if (cleanupThread != null) {
            cleanupThread.interrupt();
            try {
                cleanupThread.join(100);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            cleanupThread = null;
        }
	}

	//-------------------------------------------------------------------------
    @Override
    public void onMessage(Message jmsMsg) {
    	RT2Message rt2Msg = null;
        try {
            synchronized (receivedMessages) {
                if (receivedMessages.containsKey(jmsMsg.getJMSMessageID())) {
                    return;
                }
                receivedMessages.put(jmsMsg.getJMSMessageID(), jmsMsg.getJMSTimestamp());
            }        	
        	if (jmsMsg.getBooleanProperty(RT2MessagePostProcessor.HEADER_GPB_MSG)) {
        		BytesMessage byteMsg = (BytesMessage) jmsMsg;
        		byte [] data = new byte[(int)byteMsg.getBodyLength()];
        		byteMsg.readBytes(data);        		
        		BroadcastMessage broadcastMessage = BroadcastMessage.parseFrom(data);
        		for (BroadcastMessageReceiver broadcastMessageReceiver : broadcastMessage.getReceiversList()) {
        			rt2Msg = RT2MessageFactory.newBroadcastMessage(getRT2MessageTypeOfGpbMessage(broadcastMessage.getMsgType()), new RT2DocUidType(broadcastMessage.getDocUid().getValue()));
        			rt2Msg.setMessageID(new RT2MessageIdType(broadcastMessage.getMessageId().getValue()));
        			rt2Msg.setBodyString(broadcastMessage.getBody().getValue());
        			rt2Msg.setClientUID(new RT2CliendUidType(broadcastMessageReceiver.getReceiver().getValue()));
        			if (broadcastMessageReceiver.getSeqNr().getValue() >= 0) {
        				rt2Msg.setSeqNumber(new RT2SeqNumberType(broadcastMessageReceiver.getSeqNr().getValue()));
        			}
        			if (StringUtils.isNotEmpty(broadcastMessage.getErrorCode().getValue())) {
        				rt2Msg.setError(new RT2ErrorCodeType(getErrorCode(broadcastMessage.getErrorCode())));
        			}
        			onRT2Message(jmsMsg, rt2Msg);
        		}
        	} else {
        		rt2Msg = RT2MessageFactory.fromJmsMessage(jmsMsg);
        		onRT2Message(jmsMsg, rt2Msg);
        	}        	
        }
       catch (Exception ex) {
    	   if (rt2Msg != null) {
    		   RT2DocProxy rt2DocProxy = rt2DocRegistry.getDocProxy(rt2Msg.getClientUID(), rt2Msg.getDocUID());
    		   if (rt2DocProxy != null) {
    			   log.error(ex.getMessage() + ", msgs: " + rt2DocProxy.formatMsgsLogInfo(), ex);
    		   } else {
    			   log.error(ex.getMessage(), ex);
    		   }    		   
    	   } else {    
    		   log.error(ex.getMessage(), ex);
    	   }
       } finally {
           LogProperties.removeLogProperties();
       }
    }

	private RT2MessageType getRT2MessageTypeOfGpbMessage(MessageType msgType) {
		switch (msgType) {
			case BROADCAST_CRASHED: return RT2MessageType.BROADCAST_CRASHED;
			case BROADCAST_EDITREQUEST_STATE: return RT2MessageType.BROADCAST_EDITREQUEST_STATE;
			case BROADCAST_HANGUP: return RT2MessageType.BROADCAST_HANGUP;
			case BROADCAST_RENAMED_RELOAD: return RT2MessageType.BROADCAST_RENAMED_RELOAD;
			case BROADCAST_SHUTDOWN: return RT2MessageType.BROADCAST_SHUTDOWN;
			case BROADCAST_UPDATE: return RT2MessageType.BROADCAST_UPDATE;
			case BROADCAST_UPDATE_CLIENTS: return RT2MessageType.BROADCAST_UPDATE_CLIENTS;
			case BROADCAST_OT_RELOAD: return RT2MessageType.BROADCAST_OT_RELOAD;
			default: throw new RuntimeException("Not a broadcast message type: " + msgType);
		}
	}

	private ErrorCode getErrorCode(com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ErrorCodeType errorCodeType) {
		ErrorCode errorCode = ErrorCode.NO_ERROR;

		final String strErrorCode = errorCodeType.getValue();
		try {
			errorCode = ErrorCode.createFromJSONObject(new JSONObject(strErrorCode), ErrorCode.NO_ERROR);
		} catch (JSONException e) {
			log.error("JSONException caught trying to extract erorr code from broadcast", e);
		}
		return errorCode;
	}

    private void onRT2Message(Message jmsMsg, RT2Message rt2Msg) throws JMSException {    
    	
        docProxyResponseMetricService.startTimer(rt2Msg.getMessageID());

        if (rt2Msg.getDocUID() != null) {
            LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, rt2Msg.getDocUID().getValue());
        }
        if (rt2Msg.getClientUID() != null) {
            LogProperties.putProperty(LogProperties.Name.RT2_CLIENT_UID, rt2Msg.getClientUID().getValue());
        }

        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "processor");
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeUUID);
        LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, rt2Msg.getType().getValue());        
        
        RT2DocProxy rt2DocProxy = rt2DocRegistry.getDocProxy(rt2Msg.getClientUID(), rt2Msg.getDocUID());
        log.debug("Received ClientDoc-Msg: {}", rt2Msg.getType());

        if (rt2DocProxy == null) {
            log.debug("No doc proxy found for ClientUid: {} and DocUid: {}", rt2Msg.getClientUID(), rt2Msg.getDocUID());
            for (RT2DocProxy rt2DocProxyTmp : rt2DocRegistry.listAllDocProxies()) {
                log.debug("ClientUid: {}, DocUid: {}", rt2DocProxyTmp.getClientUID(), rt2DocProxyTmp.getDocUID());
            }
        } else {
            log.debug("Found RT2DocProxy: {}", rt2DocProxy);
            rt2DocProxy.handleResponse(rt2Msg);
        }    	
    }
    
    //-------------------------------------------------------------------------
    private class CleanupThread implements Runnable {

        @Override
        public void run() {
            boolean stop = false;
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "CleanupThread");
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeUUID);        							
            while (!stop) {
                try {
                    Thread.sleep(60000);
                    final Set<String> toRemove = new HashSet<>();
                    final long compareTime = System.currentTimeMillis() - 600000;

                    synchronized (receivedMessages) {
                        receivedMessages.forEach((k, v) -> { if (v < compareTime) toRemove.add(k);});
                        toRemove.forEach(s -> receivedMessages.remove(s));
                    }
                } catch (InterruptedException ex) {
                    stop = true;
                    Thread.currentThread().interrupt();
                } catch (Exception ex) {
                    log.error(ex.getMessage(), ex);
                }
            }
        }
    }
}
