package com.openexchange.office.rt2.core.jms;

import javax.jms.Message;
import javax.jms.MessageListener;
import javax.validation.ValidationException;

import org.apache.activemq.command.ActiveMQQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.jms.listener.DefaultMessageListenerContainer;
import org.springframework.stereotype.Service;
import org.springframework.util.backoff.ExponentialBackOff;

import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.RT2MessageSender;
import com.openexchange.office.rt2.core.cache.RT2HazelcastHelperService;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorFactory;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.core.exception.RT2InvalidClientUIDException;
import com.openexchange.office.rt2.core.metric.DocProcessorMetricService;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.annotation.ShutdownOrder;
import com.openexchange.office.tools.common.jms.JmsMessageListener;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;
import com.openexchange.office.tools.jms.PooledConnectionFactoryProxy;

@Service
@ShutdownOrder(value=-5)
public class RT2DocProcessorJmsConsumer implements MessageListener, JmsMessageListener, InitializingBean, DisposableBean {

    private static final Logger log = LoggerFactory.getLogger(RT2DocProcessorJmsConsumer.class);

    private String nodeUUID;
    
    private ActiveMQQueue queue;
    
	//--------------------------Services--------------------------------------
    @Autowired
    private RT2DocProcessorManager docProcessorManager;

    @Autowired
    private DocProcessorMetricService docProcessorMetricService;
    
    @Autowired
    private RT2DocProcessorFactory docProcessorFactory;
    
    @Autowired
    private PooledConnectionFactoryProxy pooledConnectionFactoryProxy;
    
    @Autowired
    private RT2HazelcastHelperService rt2HazelcastHelperService;
    
    @Autowired
    private RT2MessageSender messageSender;

	//------------------------------------------------------------------------    
    private DefaultMessageListenerContainer msgListenerCont;
    
	@Override
	public void afterPropertiesSet() throws Exception {
        String queueName = RT2JmsProperties.DOCUMENT_QUEUE_NAME + rt2HazelcastHelperService.getHazelcastLocalNodeUuid();
        this.queue = new ActiveMQQueue(queueName);
        log.debug("Consuming from queue with name {}", queueName);
        this.nodeUUID = rt2HazelcastHelperService.getHazelcastLocalNodeUuid();
	}

	//------------------------------------------------------------------------
	@Override
	public void startReceiveMessages() {
    	if (msgListenerCont == null) {
            msgListenerCont = new DefaultMessageListenerContainer();
            ExponentialBackOff exponentialBackOff = new ExponentialBackOff();
            exponentialBackOff.setMaxInterval(60000);
            msgListenerCont.setBackOff(exponentialBackOff);            
            msgListenerCont.setConnectionFactory(pooledConnectionFactoryProxy.getPooledConnectionFactory());
            msgListenerCont.setConcurrentConsumers(3);
            msgListenerCont.setDestination(this.queue);
            msgListenerCont.setMaxConcurrentConsumers(3);
            msgListenerCont.setPubSubDomain(false);
            msgListenerCont.setAutoStartup(true);
            msgListenerCont.setupMessageListener(this);
            msgListenerCont.setTaskExecutor(new SimpleAsyncTaskExecutor(new ThreadFactoryBuilder("RT2DocProcessorJmsConsumer-%d").build()));
            msgListenerCont.afterPropertiesSet();
            msgListenerCont.start();
    	}
	}

	//------------------------------------------------------------------------    
	@Override
	public void destroy() throws Exception {
        if (msgListenerCont != null) {
            msgListenerCont.destroy();
        }		
	}

	//------------------------------------------------------------------------    
    @Override
    public void onMessage(Message jmsMsg) {
        try {
            RT2Message rt2Msg = null;
            try {
                rt2Msg = RT2MessageFactory.fromJmsMessage(jmsMsg);
                docProcessorMetricService.startTimer(rt2Msg.getMessageID());
                LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, rt2Msg.getDocUID().getValue());
                LogProperties.putProperty(LogProperties.Name.RT2_CLIENT_UID, rt2Msg.getClientUID().getValue());
                LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "processor");
                LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeUUID);
                LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, rt2Msg.getType().getValue());
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                return;
            }

            log.debug("Received msg {}", rt2Msg);
            RT2DocProcessor rt2DocProc = docProcessorManager.getDocProcessor(rt2Msg.getDocUID());
            if (rt2DocProc == null) {
                if (RT2MessageType.REQUEST_JOIN.equals(rt2Msg.getType())) {
	                String docType = null; 
	                try {
	                	docType = RT2MessageGetSet.getDocType(rt2Msg);
	                    log.debug("Creating new docProcessor for doc with id {}", rt2Msg.getDocUID());
	                    rt2DocProc = docProcessorFactory.createInstance(docType, rt2Msg.getDocUID());
	                } catch (ValidationException ex) {
	                	log.debug("No doc processor found for request message {}", rt2Msg);
	                }
                } 
            }
            try {
            	if (rt2DocProc != null) {
            		rt2DocProc.enqueueMessage(rt2Msg);
            	}
            } catch (RT2InvalidClientUIDException ex) {
            	messageSender.sendErrorResponseToClient(rt2Msg.getClientUID(), rt2Msg, null, ex);
            	// Nothing todo, it is possible under some circumstances
            } catch (Exception ex) {
                log.error("Exception during processing RT2Message " + rt2Msg + ": " + ex.getMessage(), ex);
            	messageSender.sendErrorResponseToClient(rt2Msg.getClientUID(), rt2Msg, null, ex);
            }
        } finally {
            LogProperties.removeLogProperties();
        }
    }
}

