/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.lang.ref.WeakReference;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.text.TextStringBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.document.api.DocumentDisposer;
import com.openexchange.office.rt2.core.RT2MessageLoggerService;
import com.openexchange.office.rt2.core.RT2MessageSender;
import com.openexchange.office.rt2.core.RT2NodeInfoService;
import com.openexchange.office.rt2.core.cache.ClusterLockService;
import com.openexchange.office.rt2.core.cache.ClusterLockService.ClusterLock;
import com.openexchange.office.rt2.core.cache.RT2DocInfo;
import com.openexchange.office.rt2.core.logging.IMessagesLoggable;
import com.openexchange.office.rt2.core.logging.IMessagesObjectManager;
import com.openexchange.office.rt2.core.logging.MessagesLogger;
import com.openexchange.office.rt2.core.proxy.RT2DocInfoRegistry;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.rt2.protocol.value.RT2SessionIdType;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.annotation.ShutdownOrder;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;
import com.openexchange.office.tools.common.weakref.WeakRefUtils;

/**
 * Singleton to store all references to known RT2DocProcessor instances. The
 * references are held weak. This class should not be created, but retrieved
 * via services registry.
 *
 * @author Carsten Driesner
 * @since 7.10.0
 *
 */
@Service
@ShutdownOrder(value=-7)
@RegisteredService
public class RT2DocProcessorManager implements IDocNotificationHandler, IDocProcessorContainer, IMessagesObjectManager,
											   DisposableBean, InitializingBean {
    private static final Logger log = LoggerFactory.getLogger(RT2DocProcessorManager.class);

    //-------------------------------------------------------------------------
    private Map<RT2DocUidType, RT2DocProcessor> m_aKnownDocProcessorInstances = new HashMap<>();

    @Autowired
    private RT2MessageSender messageSender;

    @Autowired
    private DocProcessorEventService docProcessorEventService;

    @Autowired
    private DocumentDisposer documentsDisposer;

    @Autowired
    private RT2MessageLoggerService rt2MessageLoggerService;

    @Autowired
    private ClusterLockService clusterLockService;

    @Autowired
    private RT2DocInfoRegistry docInfoRegistry;

    @Autowired
    private RT2NodeInfoService nodeInfo;

    private final ScheduledExecutorService logExecutor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder("RT2DocProcessorManagerRegistryLogger-%d").build());

	//-------------------------------------------------------------------------
    @Override
	public void afterPropertiesSet() throws Exception {
    	logExecutor.scheduleAtFixedRate(new MessagesLogger(this, log), 3, 3, TimeUnit.MINUTES);
	}

    //-------------------------------------------------------------------------
    @Override
    public void destroy() throws Exception {
        logExecutor.shutdown();
        log.debug("RT2DocProcessorManager.destroy");

        Set<RT2DocUidType> keys = null;
        synchronized (m_aKnownDocProcessorInstances) {
            keys = new HashSet<>(m_aKnownDocProcessorInstances.keySet());
            m_aKnownDocProcessorInstances.clear();
        }

        keys.stream().forEach(d -> {
            disposeDocProcessor(d);
        });
    }

	//-------------------------------------------------------------------------
	@Override
	public boolean isEmpty() {
		synchronized (m_aKnownDocProcessorInstances) {
			return m_aKnownDocProcessorInstances.isEmpty();
		}
	}

    //-------------------------------------------------------------------------
	@Override
	public int size() {
		synchronized (m_aKnownDocProcessorInstances) {
			return m_aKnownDocProcessorInstances.size();
		}
	}

    //-------------------------------------------------------------------------
	@Override
	public boolean contains(RT2DocUidType rt2DocUid) {
		synchronized (m_aKnownDocProcessorInstances) {
			return m_aKnownDocProcessorInstances.containsKey(rt2DocUid);
		}
	}

    //-------------------------------------------------------------------------
	@Override
	public boolean contains(final RT2DocProcessor aDocProcessor) {
		synchronized (m_aKnownDocProcessorInstances) {
	        return m_aKnownDocProcessorInstances.containsKey(aDocProcessor.getDocUID());
		}
	}

	//-------------------------------------------------------------------------
	@Override
    public RT2DocProcessor getDocProcessor(RT2DocUidType rt2DocUid) {
		synchronized (m_aKnownDocProcessorInstances) {
			return m_aKnownDocProcessorInstances.get(rt2DocUid);
		}
    }


    //-------------------------------------------------------------------------
	@Override
	public Set<WeakReference<RT2DocProcessor>> getWeakReferenceToDocProcessors() {
		final Set<WeakReference<RT2DocProcessor>> aResult = new HashSet<>();
		synchronized (m_aKnownDocProcessorInstances) {
		    m_aKnownDocProcessorInstances.values().forEach(proc -> aResult.add(new WeakReference<RT2DocProcessor>(proc)));
		}
		return aResult;
	}

    //-------------------------------------------------------------------------
	public Set<RT2DocProcessor> getDocProcessors() {
		final Set<RT2DocProcessor> aResult = new HashSet<>();
		synchronized (m_aKnownDocProcessorInstances) {
		    m_aKnownDocProcessorInstances.values().forEach(proc -> aResult.add(proc));
		}
		return aResult;
	}

    //-------------------------------------------------------------------------
	@Override
	public void docProcessorCreated(final RT2DocProcessor aCreatedInstance) {
		synchronized (m_aKnownDocProcessorInstances) {
			m_aKnownDocProcessorInstances.put(aCreatedInstance.getDocUID(), aCreatedInstance);
		}

		documentsDisposer.addDocumentRef(aCreatedInstance);
		docProcessorEventService.notifyDocCreated(aCreatedInstance.getDocUID());
	}

    //-------------------------------------------------------------------------
	@Override
	public void docProcessorDisposed(final RT2DocProcessor aDisposedInstance) {
	    if (aDisposedInstance != null) {
    		final RT2DocUidType sDocUID = aDisposedInstance.getDocUID();

    		synchronized (m_aKnownDocProcessorInstances) {
    			if (m_aKnownDocProcessorInstances.containsKey(sDocUID)) {
	        		log.debug("Removing docProcessor with docUid ''{}''", sDocUID);
	        		m_aKnownDocProcessorInstances.remove(sDocUID);
	        		documentsDisposer.removeDocumentRef(sDocUID.getValue());
	        		List<String> logMsgs = rt2MessageLoggerService.formatMsgsLogInfo(aDisposedInstance.getDocUID());
	        		if (!logMsgs.isEmpty()) {
	        			TextStringBuilder builder = new TextStringBuilder(logMsgs.get(0));
	        			logMsgs.remove(0);
	        			logMsgs.forEach(s -> builder.appendln("    " + s));
	        			log.info("Messages of disposed docprocessor with docUid {}: {}", aDisposedInstance.getDocUID(), builder);
	        		} else {
	        			log.info("Messages of disposed docprocessor with docUid {}", aDisposedInstance.getDocUID());
	        		}
	        		docProcessorEventService.notifyDocDisposed(aDisposedInstance.getDocUID());
    			}
    		}
	    }
	}

    //-------------------------------------------------------------------------
    public void preShutdown() {
        final Set<WeakReference<RT2DocProcessor>> knownDocProcessors = getWeakReferenceToDocProcessors();
        final Iterator<WeakReference<RT2DocProcessor>> iter = knownDocProcessors.iterator();

        while (iter.hasNext()) {
            RT2DocUidType docUID = null;

            try {
                final RT2DocProcessor docProcessor = WeakRefUtils.getHardRef(iter.next());
                if (docProcessor != null) {
                    docUID = docProcessor.getDocUID();
                    final RT2Message shutdownBroadcastMsg = RT2MessageFactory.newBroadcastMessage(RT2MessageType.BROADCAST_SHUTDOWN, docUID);
                    shutdownBroadcastMsg.setDocUID(docUID);
                    messageSender.broadcastMessageExceptClient(null, shutdownBroadcastMsg, RT2MessageType.BROADCAST_SHUTDOWN);
                }
            } catch (Exception e) {
                log.error("Exception caught trying to send clients of document {} a BROADCAST_SHUTDOWN message", ((docUID == null) ? "null" : docUID.getValue()), e);
            }
        }
    }

    //-------------------------------------------------------------------------
	@Override
	public Collection<IMessagesLoggable> getManagedCollection() {
		return new HashSet<>(m_aKnownDocProcessorInstances.values());
	}

    //-------------------------------------------------------------------------
    public Set<RT2DocProcessor> getDocProcessorsAssociatedToSessionId(RT2SessionIdType sessionId) {
    	Set<RT2DocProcessor> res = new HashSet<>();
    	for (RT2DocProcessor docProcessor : m_aKnownDocProcessorInstances.values()) {
    		if (docProcessor.getSessionIdToClientUids().containsKey(sessionId)) {
    			res.add(docProcessor);
    		}
    	}
    	return res;
    }

    //-------------------------------------------------------------------------
    private void disposeDocProcessor(RT2DocUidType docUid) {
        final RT2DocInfo docInfo = docInfoRegistry.getDocInfo(docUid);

        final ClusterLock clusterLock = clusterLockService.getLock(docUid);
        boolean locked = false;
        try {
            locked = clusterLock.lock();
            if (locked) {
                log.debug("tryLock for doc-uid {}", docUid);
                disposeAndFreeDocProcessorResources(docUid, docInfo);
            } else {
                log.warn("Lock couldn't acquired in for doc-uid {}", docUid);
            }
        } catch (Throwable ex) {
            ExceptionUtils.handleThrowable(ex);
            log.error(ex.getMessage(), ex);
        } finally {
            if (locked) {
                clusterLock.unlock();
            }
        }
    }

    //-------------------------------------------------------------------------
    private void disposeAndFreeDocProcessorResources(RT2DocUidType docUid, RT2DocInfo docInfo) {
        final RT2DocProcessor docProcessor = getDocProcessor(docUid);
        if (docProcessor != null) {
            docProcessor.dispose();
        }
        nodeInfo.deregisterDocOnNode(docUid);
        if (docInfo != null) {
            docInfo.destroyRefCount4Clients();
        }
        docInfoRegistry.freeDocInfos(docUid);
    }
}
