/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.rt2.core.exception.RT2Exception;
import com.openexchange.office.rt2.core.exception.RT2InvalidClientUIDException;
import com.openexchange.office.rt2.core.exception.RT2TypedException;
import com.openexchange.office.rt2.core.sequence.SequenceDocProcessor;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.user.LoadState;
import com.openexchange.office.tools.common.user.UserData;
import com.openexchange.office.tools.common.user.UserDataManager;
import com.openexchange.office.tools.common.user.UserHelper;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.user.User;

//=============================================================================
public abstract class DocProcessor extends SequenceDocProcessor
{
    //-------------------------------------------------------------------------
	private static final Logger log = LoggerFactory.getLogger(DocProcessor.class);

    //-------------------------------------------------------------------------
	private final UserDataManager m_aUserData       = new UserDataManager();

    //-------------------------------------------------------------------------
    private final ClientsStatus   m_aClientsStatus;

    //-------------------------------------------------------------------------
    private final DocumentStatus  m_aDocumentStatus;

    //-------------------------------------------------------------------------
    private final Object          m_aStatusSync     = new Object();

    //-------------------------------------------------------------------------
	public DocProcessor() throws Exception
	{
		super();

		m_aDocumentStatus = createDocumentStatus();
		m_aClientsStatus  = createClientsStatus();
	}

    //-------------------------------------------------------------------------
	protected Object getStatusSyncObject()
	{
		return m_aStatusSync;
	}

    //-------------------------------------------------------------------------
	public UserData getUserData(final RT2CliendUidType sClientUID)
	{
		return m_aUserData.getUserData(sClientUID.getValue());
	}

    //-------------------------------------------------------------------------
	@Override
	public List<UserData> getUserDatasCloned()
	{
		return m_aUserData.getUserDataCloned();
	}

    //-------------------------------------------------------------------------
	public UserData getUserDataThrowIfUnknown(final RT2CliendUidType sClientUID) throws RT2Exception {
		final UserData userData = getUserData(sClientUID);
		if (userData == null) {
            final String logMessage = "no user data for client uid [" + sClientUID + "]";
            final RT2TypedException ex = new RT2InvalidClientUIDException(new ArrayList<>(), "details", logMessage);
            throw ex;
		}
		return userData;
	}

    //-------------------------------------------------------------------------
	public UserData getMemberUserData(final String sClientUID)
	{
		return m_aUserData.getMemberUserData(sClientUID);
	}

    //-------------------------------------------------------------------------
	public List<String> getUserIDsFiltered(final Predicate<UserData> aFilter)
	{
		List<String> aFilteredList = m_aUserData.getFilteredUserData(aFilter);
		return aFilteredList;
	}

    //-------------------------------------------------------------------------
	public ClientsStatus getClientsStatus()
	{
		return m_aClientsStatus;
	}

    //-------------------------------------------------------------------------
	public ClientsStatus getClientsStatusClone()
	{
		synchronized (m_aStatusSync)
		{
			try {
				return m_aClientsStatus.clone();
			} catch (CloneNotSupportedException e) {
				// Should never be happened
				throw new RuntimeException(e);
			}
		}
	}

    //-------------------------------------------------------------------------
	public ClientsStatus getClientsStatusCloneIfModified(boolean bResetModifiedState) throws CloneNotSupportedException
	{
		ClientsStatus result = null;

		synchronized (m_aStatusSync)
		{
			if (m_aClientsStatus.isModified())
			{
				result = m_aClientsStatus.clone();
				if (bResetModifiedState)
					m_aClientsStatus.clearModifications();
			}
		}

		return result;
	}

    //-------------------------------------------------------------------------
	public DocumentStatus getDocumentStatus()
	{
		return m_aDocumentStatus;
	}

    //-------------------------------------------------------------------------
	public DocumentStatus getDocumentStatusClone() throws CloneNotSupportedException
	{
		synchronized (m_aStatusSync)
		{
			return m_aDocumentStatus.clone();
		}
	}

    //-------------------------------------------------------------------------
	public DocumentStatus getDocumentStatusCloneIfModified(boolean bResetModifiedState) throws CloneNotSupportedException
	{
		DocumentStatus result = null;

		synchronized (m_aStatusSync)
		{
			if (m_aDocumentStatus.isModified())
			{
				result = m_aDocumentStatus.clone();
				if (bResetModifiedState)
					m_aDocumentStatus.setModified(false);
			}
		}

		return result;
	}

    //-------------------------------------------------------------------------
	@Override
	public List<DocProcessorClientInfo> getClientsInfo()
	{
		List<DocProcessorClientInfo> aClientsInfo = new ArrayList<>();

		final List<UserData> aUserDataList = m_aUserData.getUserData();
		for (final UserData aUserData : aUserDataList) {
			aClientsInfo.add(new DocProcessorClientInfo(new RT2CliendUidType(aUserData.getUserId()), aUserData.getNodeUUID()));
		}
		return aClientsInfo;
	}

	//-------------------------------------------------------------------------
    @Override
	public void updateClientsInfo(Collection<RT2CliendUidType> existingClients) {
    	m_aUserData.updateUserData(existingClients.stream().map(c -> c.getValue()).collect(Collectors.toSet()));
	}

	//-------------------------------------------------------------------------
    @Override
    public boolean onJoin(final RT2Message aRequest) throws Exception
    {
        final String sFolderID    = RT2MessageGetSet.getFolderID (aRequest);
        final String sFileID      = RT2MessageGetSet.getFileID   (aRequest);
        final String sSessionID   = RT2MessageGetSet.getSessionID(aRequest);
        final String sNodeUID     = RT2MessageGetSet.getNodeUUID (aRequest);

        addUser(aRequest.getClientUID(), sFolderID, sFileID, sSessionID, sNodeUID);

        return true;
    }

    //-------------------------------------------------------------------------
    @Override
    public boolean onLeave(final RT2Message aRequest) throws Exception
    {
        final RT2Message aResponse  = RT2MessageFactory.createResponseFromMessage(aRequest, RT2MessageType.RESPONSE_LEAVE);

        messageSender.sendMessageWOSeqNrTo (aRequest.getClientUID(), aResponse);

        removeUser(aRequest.getClientUID());

        return true;
    }

    //-------------------------------------------------------------------------
    @Override
    public boolean onEmergencyLeave(final RT2Message aRequest) throws Exception
    {
        final RT2Message aResponse  = RT2MessageFactory.createResponseFromMessage(aRequest, RT2MessageType.RESPONSE_EMERGENCY_LEAVE);

        messageSender.sendMessageWOSeqNrTo (aRequest.getClientUID(), aResponse);

        removeUser(aRequest.getClientUID());

        return true;
    }


    //-------------------------------------------------------------------------
	protected int getUserCount()
	{
		return m_aUserData.getCount();
	}

    //-------------------------------------------------------------------------
	public void addUser(RT2CliendUidType sClientUID, String sFolderID, String sFileID, String sSessionID, String sNodeUUID)
	    throws Exception
	{
        synchronized(m_aUserData)
        {
            UserData aUserData  = m_aUserData.getUserData(sClientUID.getValue());
            if (aUserData == null)
            {
                aUserData  = new UserData(sClientUID.getValue(), sFolderID, sFileID, sSessionID, false, null, sNodeUUID);
                m_aUserData.addUser(sClientUID.getValue(), aUserData);
            }
        }

        final ServerSession aSession = sessionService.getSession4Id(sSessionID);
        if (aSession == null) {
        	throw new RT2TypedException(ErrorCode.GENERAL_SESSION_INVALID_ERROR, rt2MessageLoggerService.getMsgsAsList(getDocUID()));
        }
        final User   aUserInfo    = aSession.getUser();
        final String sDisplayName = UserHelper.getFullName(aUserInfo);

        synchronized (m_aStatusSync)
        {
            m_aClientsStatus.addClient(sClientUID, sDisplayName, aUserInfo.getId(), aUserInfo.isGuest(), null);
        }
	}

    //-------------------------------------------------------------------------
	protected void updateUserData(String sClientUID, String sEncryptionInfo)
        throws Exception
	{
        Validate.notEmpty(sClientUID, "invalid argument 'clientUID'.");

        boolean bUserFound = false;

        synchronized(m_aUserData)
        {
            final UserData aUserData  = m_aUserData.getUserData(sClientUID);
            if (aUserData != null)
            {
                bUserFound = true;
                aUserData.setEncryptionInfo(sEncryptionInfo);
            }
        }

        if (!bUserFound)
            log.warn("Unknown client UID: {}", sClientUID);
    }

    //-------------------------------------------------------------------------
	protected void updateUserData(RT2CliendUidType sClientUID, LoadState eLoadState)
        throws Exception
	{
        Validate.notEmpty(sClientUID.getValue(), "invalid argument 'clientUID'.");

        boolean bUserFound = false;

        final UserData aUserData  = m_aUserData.getUserData(sClientUID.getValue());
        if (aUserData != null) {
            bUserFound = true;
            aUserData.setLoadState(eLoadState);
        }

        if (!bUserFound)
            log.warn("Unknown client UID: {}", sClientUID);
	}

    //-------------------------------------------------------------------------
    public void removeUser(RT2CliendUidType sClientUID)
    {
        m_aUserData.removeUser(sClientUID.getValue());

        synchronized (m_aStatusSync)
        {
        	m_aClientsStatus.removeClient(sClientUID);
        }
    }

    //-------------------------------------------------------------------------
    protected abstract DocumentStatus createDocumentStatus() throws Exception;

    //-------------------------------------------------------------------------
    protected abstract ClientsStatus createClientsStatus() throws Exception;
}
