/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rt2.core;

import java.lang.ref.WeakReference;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.Sets;
import com.hazelcast.core.DistributedObject;
import com.hazelcast.core.DistributedObjectUtil;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IAtomicLong;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.exception.OXException;
import com.openexchange.java.ConcurrentHashSet;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.cache.ClusterLockException;
import com.openexchange.office.rt2.core.cache.ClusterLockService;
import com.openexchange.office.rt2.core.cache.ClusterLockService.ClusterLock;
import com.openexchange.office.rt2.core.cache.RT2DocInfo;
import com.openexchange.office.rt2.core.cache.RT2HazelcastHelperService;
import com.openexchange.office.rt2.core.config.RT2ConfigService;
import com.openexchange.office.rt2.core.doc.RT2DocProcessor;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.core.proxy.EDocProxyState;
import com.openexchange.office.rt2.core.proxy.RT2DocInfoRegistry;
import com.openexchange.office.rt2.core.proxy.RT2DocProxy;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyRegistry;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealth;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealthMap;
import com.openexchange.office.rt2.hazelcast.RT2NodeHealthState;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeHealthState;
//import com.openexchange.office.rt2.protocol.internal.Cause;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2NodeUuidType;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.annotation.ShutdownOrder;
import com.openexchange.office.tools.common.log.LogMethodCallHelper;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;
import com.openexchange.office.tools.common.weakref.WeakRefUtils;

@Service
@ShutdownOrder(value=-9)
@RegisteredService
public class RT2GarbageCollector implements Runnable, DisposableBean, InitializingBean {

	private static final Logger log = LoggerFactory.getLogger(RT2GarbageCollector.class);
	public static final String REFCOUNT_CLIENT_PREFIX  = "rt2cache.atomic.";
	public static final String REFCOUNT_CLIENT_POSTFIX = ".refcount.clients.a";

    //-------------------------------------------------------------------------
	public static final String KEY_GC_FREQUENCY = "rt2.gc.frequency";

    //---------------------------Services--------------------------------------
	@Autowired
	private RT2DocProcessorManager docProcMngr;

	@Autowired
	private ClusterLockService clusterLockService;

	@Autowired
	private RT2DocInfoRegistry docInfoRegistry;

	@Autowired
	private RT2DocProxyRegistry docProxyRegistry;

	@Autowired
	private RT2DocOnNodeMap docOnNodeMap;;

	@Autowired
	private RT2ConfigService configService;

	@Autowired
	private MetricRegistry metricRegistry;

	@Autowired
	private HazelcastInstance hazelcastInstance;

	@Autowired
	private RT2HazelcastHelperService rt2HazelcastHelperService;

    @Autowired
    private RT2NodeInfoService rt2NodeInfo;

    @Autowired
    private RT2NodeHealthMap nodeHealthMap;

    //-------------------------------------------------------------------------

	private final ScheduledExecutorService finalRt2DocProcessorRemoveScheduler = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder("RT2GarbageCollectorFinalProcRemove-%d").build());

	private ScheduledExecutorService m_aScheduler = null;

	private Counter docProxyOfflineCounter;
	private Counter docProxyRemovedCounter;
	private Counter docProcRemovedCounter;
	private Counter refCountClientsRemovedCounter;
	private Counter docOnNodeMapEntryRemovedCounter;

	private Pattern docUidOfAtomicLongNamePattern = Pattern.compile("rt2cache.atomic.(.*).refcount.clients.a");

	private Map<RT2DocUidType, LocalDateTime> atomicLongsToGc = new HashMap<>();
	private Set<RT2DocUidType> atomicLongsToBeVerified = new ConcurrentHashSet<>();
	private Set<RT2DocUidType> docUidsToClear = new ConcurrentHashSet<>();
	private Map<RT2NodeUuidType, LocalDateTime> docOnNodeEntriesToGC = new ConcurrentHashMap<>();
	private long minWaitTimeBeforeGcAtomicLong = 5 * 60; // 5 minutes in seconds

	public RT2GarbageCollector() {
	}

	public RT2GarbageCollector(long minWaitTimeBeforeGcAtomicLong) {
		this.minWaitTimeBeforeGcAtomicLong = minWaitTimeBeforeGcAtomicLong;
	}

	//-------------------------------------------------------------------------
    public Map<RT2DocUidType, LocalDateTime> getAtomicLongsToGc() {
		return new HashMap<>(atomicLongsToGc);
	}

	//-------------------------------------------------------------------------
	public Set<RT2DocUidType> getAtomicLongsToBeVerified() {
		return new HashSet<>(atomicLongsToBeVerified);
	}

    @Override
	public void afterPropertiesSet() throws Exception {

	    if (m_aScheduler != null)
	        return;

		this.docProxyOfflineCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docProxy", "offline"));
		this.docProxyRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docProxy", "removed"));
		this.docProcRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docProcessor", "removed"));
		this.refCountClientsRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "refCountClients", "removed"));
		this.docOnNodeMapEntryRemovedCounter = metricRegistry.counter(MetricRegistry.name("GarbageCollector", "docOnNodeMap", "removed"));

	    LogMethodCallHelper.logMethodCall(log, this, "start");

        final Runnable aRunner = () -> {impl_doGC ();};

        final long                     nFrequencyInMS = configService.getRT2GCFrequencyInMS();
        if (nFrequencyInMS > 0) {
	        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("RT2GarbageCollector-%d").build();
	        final ScheduledExecutorService aScheduler     = Executors.newScheduledThreadPool(1, namedThreadFactory);
	        aScheduler.scheduleAtFixedRate(aRunner, nFrequencyInMS, nFrequencyInMS, TimeUnit.MILLISECONDS);
	        m_aScheduler = aScheduler;
        }
        LogMethodCallHelper.logMethodCallRes(log, this.getClass(), "start", Void.class);
	}

	@Override
	public void destroy() throws Exception {
        if (m_aScheduler == null)
            return;

        final ScheduledExecutorService aScheduler = m_aScheduler;
        m_aScheduler = null;
        LogMethodCallHelper.logMethodCall(log, this, "stop");

        aScheduler.shutdownNow();
        finalRt2DocProcessorRemoveScheduler.shutdown();

        try {
	        final boolean bOK = aScheduler.awaitTermination(30000, TimeUnit.MILLISECONDS);
	        if ( ! bOK) {
	            log.warn("... RT2-GC not stopped in time !");
	            // no exception please - GC is stopped in shutdown of process ...
	            // so thread will be stopped anyway ...
	        } else {
	            LogMethodCallHelper.logMethodCallRes(log, this.getClass(), "stop", Void.class);
	        }
        } catch (InterruptedException ex) {
        	Thread.currentThread().interrupt();
        }
    }

	//-------------------------------------------------------------------------
	@Override
    public void run ()
	{
	    LogMethodCallHelper.logMethodCall(log, this, "start");

        impl_doGC ();
        LogMethodCallHelper.logMethodCallRes(log, this.getClass(), "start", Void.class);
	}

    //-------------------------------------------------------------------------

	public boolean doGcForDocUid(RT2DocUidType docUid, boolean rememberWhenException) {
        final ClusterLock clusterLock = clusterLockService.getLock(docUid);
        boolean locked = false;
        boolean lockExceptionOccured = false;
        try {
            log.debug("Lock called by garbage collector for doc with id [{}]", docUid);
            locked = clusterLock.lock();
            rt2NodeInfo.deregisterDocOnNode(docUid);
            final RT2DocInfo aDocInfo = docInfoRegistry.getDocInfo(docUid);
            aDocInfo.destroyRefCount4Clients();
            return true;
        } catch (ClusterLockException e) {
        	lockExceptionOccured = true;
		} finally {
        	if (locked) {
        		clusterLock.unlock();
        	}
		}
        if (rememberWhenException && (!locked || lockExceptionOccured)) {
        	docUidsToClear.add(docUid);
        }
        return false;
	}

    //-------------------------------------------------------------------------

    public void doGC() {
    	impl_doGC();
    }

    //-------------------------------------------------------------------------
    public void registerDocRefCountForRemovalVerification(RT2DocUidType refCountOfdocUIDToCheck) {
        atomicLongsToBeVerified.add(refCountOfdocUIDToCheck);
    }

    //-------------------------------------------------------------------------
    private void impl_doGC () {
        try {
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2GarbageCollector");
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, rt2HazelcastHelperService.getHazelcastLocalNodeUuid());
            checkAndGCOfflineOrStaleDocProxies();
            checkAndGCStaleDocResources();
            checkAndGCAtomicLongAsClientRef();
            checkAndEnsureConsitencyOfDocOnNodeMapForThisNode();
            checkDetectedDocUids();
            checkAndEnsureConsistencyOfDocOnNodeMapGlobally();
        } catch (Exception ex) {
            log.error(ex.getMessage(), ex);
        }
    }

    //-------------------------------------------------------------------------
    private void checkDetectedDocUids() {
		final Set<RT2DocUidType> toRemove = new HashSet<>();
		docUidsToClear.forEach(docUid -> {
			if (doGcForDocUid(docUid, false)) {
				toRemove.add(docUid);
			}
		});
		docUidsToClear.removeAll(toRemove);
	}

    //-------------------------------------------------------------------------
    private /* no synchronized */ void checkAndGCOfflineOrStaleDocProxies () {
        LogMethodCallHelper.logMethodCall(log, this, "checkAndGCOfflineOrStaleDocProxies");

        final RT2DocProxyRegistry aDocRegistry         = docProxyRegistry;
        final List< RT2DocProxy > lDocProxies          = aDocRegistry.listAllDocProxies();
        final long                nOfflineTresholdInMS = configService.getRT2GCOfflineTresholdInMS();
        final long                nNow                 = System.currentTimeMillis();

        for (final RT2DocProxy aDocProxy : lDocProxies) {
            final long nOfflineSinceInMS  = aDocProxy.getOfflineTimeInMS();
            final boolean bOfflineTimeout = (nOfflineSinceInMS > nOfflineTresholdInMS);
            final boolean bStaleDocProxy  = isStaleDocProxy(aDocProxy, nNow, nOfflineTresholdInMS);

            if (bOfflineTimeout || bStaleDocProxy) {
                if (bOfflineTimeout) {
                	docProxyOfflineCounter.inc();
                    final String sOfflineSinceHumanReadable = DurationFormatUtils.formatDuration(nOfflineSinceInMS, "HH:mm:ss", /* pad with zeros */ true);
                    log.debug("Doc proxy {} is offline for {}", aDocProxy, sOfflineSinceHumanReadable);
                } else {
                    final long nStaleDuration = Math.abs(nNow - aDocProxy.getCreationTimeInMS());
                    final String sStaleSinceHumanReadable = DurationFormatUtils.formatDuration(nStaleDuration, "HH:mm:ss", /* pad with zeros */ true);
                    log.debug("Doc proxy {} is stale for {}", aDocProxy, sStaleSinceHumanReadable);
                }

	            if (!bStaleDocProxy) {
	            	try {
	            		aDocProxy.closeHard(false);
	            	} catch (Exception ex) {
	            		// An exception here is not a problem because there is an additional task(checkAndGCStaleDocResources()) which removes the corresponding RT2DocProcessor
	            		log.debug("GC failed for docProxy {}", aDocProxy, ex);
	            	}
	            }
	            docProxyRemovedCounter.inc();
                aDocRegistry.deregisterDocProxy(aDocProxy, true);
            }
        }
    }

    //-------------------------------------------------------------------------
    private /* no synchronized */ boolean isStaleDocProxy(final RT2DocProxy aDocProxy, long nNow, long nOfflineTresholdInMS) {
        Validate.notNull(aDocProxy);

        final long nCreationTime           = aDocProxy.getCreationTimeInMS();
        final EDocProxyState docProxyState = aDocProxy.getDocState();

        return ((docProxyState == EDocProxyState.E_INIT) && (Math.abs(nNow - nCreationTime) > nOfflineTresholdInMS));
    }

    //-------------------------------------------------------------------------
    private /* no synchronized */ void checkAndGCStaleDocResources ()
    {
        LogMethodCallHelper.logMethodCall(log, this, "checkAndGCStaleDocResources");
        final Set<WeakReference<RT2DocProcessor>> aLocalDocProcessors = docProcMngr.getWeakReferenceToDocProcessors();

        if (!aLocalDocProcessors.isEmpty())
        {
            final Set<RT2DocProcessor> aDocProcessorsWOClients = retrieveDocProcessorsWithoutClients(aLocalDocProcessors);
            final RT2DocInfoRegistry       aDocRegistry        = docInfoRegistry;
            Set<RT2DocUidType> possibleDocProcessorToRemove = new HashSet<>();
            for (final RT2DocProcessor aDocProc : aDocProcessorsWOClients)
            {
                final RT2DocUidType     sDocUID   = aDocProc.getDocUID();
                final RT2DocInfo aDocInfo  = aDocRegistry.peekDocInfo(sDocUID);
               	destroyRefCount4Clients(sDocUID, aDocInfo);
                possibleDocProcessorToRemove.add(sDocUID);
            }
            if (!possibleDocProcessorToRemove.isEmpty()) {
            	finalRt2DocProcessorRemoveScheduler.schedule(new FinalRt2DocProcessorRemoveThread(possibleDocProcessorToRemove), 5, TimeUnit.MINUTES);
            }
        }
    }

    //-------------------------------------------------------------------------
	private void destroyRefCount4Clients(final RT2DocUidType docUID, final RT2DocInfo docInfo) {
		final ClusterLock clusterLock = clusterLockService.getLock(docUID);
		boolean locked = false;
		try {
			locked = clusterLock.lock();
			if (locked) {
				docInfo.destroyRefCount4Clients();
			}
		} catch (ClusterLockException e) {
			// Ignore, try again next time
		} finally {
			if (locked) {
		        clusterLock.unlock();
			}
		}
	}

    //-------------------------------------------------------------------------
    void checkAndGCAtomicLongAsClientRef ()
    {
        final HazelcastInstance aHzCore = hazelcastInstance;
        if (null != aHzCore) {
        	final Collection<DistributedObject> aDistributedObjects = aHzCore.getDistributedObjects();
            final Set<DistributedObject> aClientRefCountSet  =
            		aDistributedObjects.stream()
                                       .filter(RT2GarbageCollector::isAtomicLong)
                                       .filter(RT2GarbageCollector::isClientRefCountObject)
                                       .collect(Collectors.toSet());

            // check distributed object list for gc-able ref-counts
            for (final DistributedObject aObject : aClientRefCountSet)
            {
                try
                {
                    checkAndDestroyAtomicLongSafely((IAtomicLong)aObject);
                }
                catch (Throwable t)
                {
                    ExceptionUtils.handleThrowable(t);
                    String distributeObjectname = ((null == aObject) ? "null" : aObject.getName());
                    log.error("RT2: GC failed for client ref-count instance " + distributeObjectname  + " found with count = 0", t);
                }
            }

            // insert the registered atomic long names in the gc map
            for (final RT2DocUidType docUIDRefCount : atomicLongsToBeVerified) {
                final String hzAtomicLongName = RT2DocInfo.generateNameFromDocUid(docUIDRefCount.getValue());
                IAtomicLong hzRefCountForDoc = aHzCore.getAtomicLong(hzAtomicLongName);

                try {
                    checkAndDestroyAtomicLongSafely(hzRefCountForDoc);
                } catch (Throwable t) {
                    ExceptionUtils.handleThrowable(t);
                    log.error("RT2: GC failed for client ref-count instance " + hzAtomicLongName  + " found with count = 0", t);
                }
            }
        }
    }

    //-------------------------------------------------------------------------
    public static boolean isClientRefCountObject(final DistributedObject distributedObject)
    {
        return DistributedObjectUtil.getName(distributedObject).startsWith(REFCOUNT_CLIENT_PREFIX);
    }

    //-------------------------------------------------------------------------
    public static boolean isAtomicLong(final DistributedObject distributedObject)
    {
        return (distributedObject instanceof IAtomicLong);
    }

    //-------------------------------------------------------------------------
    private void checkAndDestroyAtomicLongSafely(IAtomicLong aClientRefCount)
    {
    	final String docUID = getDocUIDFromClientRefCountName(aClientRefCount);
        if (StringUtils.isNotEmpty(docUID) && shouldBeVerifiedOrGarbageCollected(aClientRefCount)) {
        	RT2DocUidType docUidObj = new RT2DocUidType(docUID);
        	ClusterLock clusterLock = clusterLockService.getLock(new RT2DocUidType(docUID));
            try {
            	clusterLock.lock();
                long refCount = aClientRefCount.get();
                boolean bCanBeGC = (refCount == 0);
                if (!bCanBeGC && ((refCount >= RT2Constants.REF_COUNT_TRESHHOLD) || refCount < 0)) {
                	if (atomicLongsToGc.containsKey(docUidObj)) {
                		LocalDateTime insertTime = atomicLongsToGc.get(docUidObj);
                		if (insertTime.plusSeconds(minWaitTimeBeforeGcAtomicLong).isBefore(LocalDateTime.now())) {
                			bCanBeGC = true;
                			atomicLongsToGc.remove(docUidObj);
                            atomicLongsToBeVerified.remove(docUidObj);
                		}
                	} else {
                		atomicLongsToGc.put(docUidObj, LocalDateTime.now());
                	}
                } else {
                	atomicLongsToGc.remove(docUidObj);
                    atomicLongsToBeVerified.remove(docUidObj);
                }
                if (bCanBeGC) {
					Matcher m = docUidOfAtomicLongNamePattern.matcher(aClientRefCount.getName());
					if (m.find()) {
						LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, m.group(1));
						log.info("Removing AtomicLong with name {}", aClientRefCount.getName());
					}
                	refCountClientsRemovedCounter.inc();
                	aClientRefCount.destroy();
                }
        	} catch (ClusterLockException e) {
        		// Ignore, try again next time
        	} finally {
            	clusterLock.unlock();
				LogProperties.remove(LogProperties.Name.RT2_DOC_UID);
            }
        } else {
        	// verification completed, remove entry as we see a normal value for the ref-count
        	if (docUID != null) {
	        	RT2DocUidType docUidObj = new RT2DocUidType(docUID);
	            atomicLongsToBeVerified.remove(docUidObj);
	            atomicLongsToGc.remove(docUidObj);
        	}
        }
    }

    //-------------------------------------------------------------------------
    private static boolean shouldBeVerifiedOrGarbageCollected(IAtomicLong atomicLong) {
        long refCount = atomicLong.get();
        return (refCount <= 0) || (refCount >= RT2Constants.REF_COUNT_TRESHHOLD);
    }

    //-------------------------------------------------------------------------
    private static String getDocUIDFromClientRefCountName(final IAtomicLong aClientRefCount)
    {
        String sResult = null;
        final String sName  = aClientRefCount.getName();
        final int    nStart = REFCOUNT_CLIENT_PREFIX.length();
        final int    nEnd   = sName.indexOf(".", nStart);

        if ((nEnd > nStart) && (nEnd < sName.length()))
            sResult = sName.substring(nStart, nEnd);

        return sResult;
    }

    //-------------------------------------------------------------------------
    private void checkAndEnsureConsitencyOfDocOnNodeMapForThisNode() {
        LogMethodCallHelper.logMethodCall(log, this, "checkAndEnsureConsitencyOfDocOnNodeMapForThisNode");

        RT2DocOnNodeMap aDocOnNodeMap = docOnNodeMap;
        Set<String> docIds = aDocOnNodeMap.getDocsOfMember();
        for (String str : docIds) {
            RT2DocUidType docUid = new RT2DocUidType(str);
            if (!docProcMngr.contains(docUid)) {
                LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, docUid.getValue());
                removeDocOnNodeEntryAndAtomicLongForDocUid(docUid, false);
            }
        }
    }

    //-------------------------------------------------------------------------
    private void checkAndEnsureConsistencyOfDocOnNodeMapGlobally() {
        LogMethodCallHelper.logMethodCall(log, this, "checkAndEnsureConsistencyOfDocOnNodeMapGlobally");

        RT2DocOnNodeMap aDocOnNodeMap = docOnNodeMap;
        final Set<Member> currentMembers = hazelcastInstance.getCluster().getMembers();
        final Set<String> currentMemberUUIDs = currentMembers.stream().map(m -> m.getUuid()).collect(Collectors.toSet());
        final Set<String> memberUUIDsWithDocs = aDocOnNodeMap.getMember();
        if (memberUUIDsWithDocs != null) {
            final Set<String> unknownMembersWithDocs = Sets.difference(memberUUIDsWithDocs, currentMemberUUIDs);
            unknownMembersWithDocs.stream().forEach(m -> {
                try {
                    final UUID cleanupNode = UUID.fromString(m);
                    final RT2NodeUuidType cleanupNodeUUID = new RT2NodeUuidType(cleanupNode);

                    if (tryLockForMember(cleanupNodeUUID)) {
                        log.info("Removing docOnNodeMap entries for unknown member node-uuid {}", m);
                        final Set<String> docsOnUnknownNode = aDocOnNodeMap.getDocsOfMember(m);
                        docsOnUnknownNode.stream().forEach(d -> {
                            RT2DocUidType docUid = new RT2DocUidType(d);
                            LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, docUid.getValue());
                            removeDocOnNodeEntryAndAtomicLongForDocUid(docUid, true);
                        });
                        removeCleanedUpMemberFromHealthMap(cleanupNodeUUID);
                    }
                } catch (IllegalArgumentException e) {
                    log.error("Node uuid detected in DocOnNodeMap with illegal uuid " + m, e);
                } catch (Exception e) {
                    log.error("Exception caught trying to clean-up documents for unknown member node-uuid " + m, e);
                }
            });
            // in case there are no unknown members we clear our map to remove obsolete entries
            if (unknownMembersWithDocs.isEmpty()) {
                docOnNodeEntriesToGC.clear();
            }
        }
    }

    //-------------------------------------------------------------------------
    private boolean tryLockForMember(RT2NodeUuidType nodeUuid) {
        boolean result = false;

        try {
            boolean locked = false;
            if (shouldThisNodeTryToTakeOverOwnershipForNotMemberNode(nodeUuid)) {
                RT2NodeHealthState nodeHealthState = nodeHealthMap.get(nodeUuid.toString());
                final String nodeHealthMapName = nodeHealthMap.getUniqueMapName();
                final IMap<String, PortableNodeHealthState> hzNodeHealthMap = hazelcastInstance.getMap(nodeHealthMapName);

                try {
                    int nRetryCount = 2;
                    while ((nRetryCount > 0) && !locked) {
                        locked = hzNodeHealthMap.tryLock(nodeUuid.getValue().toString(), 1000, TimeUnit.MILLISECONDS);

                        if (locked) {
                            final String thisMemberUuidString = hazelcastInstance.getCluster().getLocalMember().getUuid();
                            final String cleanupUUID = (null != nodeHealthState) ? nodeHealthState.getCleanupUUID(): null;
                            if (nodeHealthState == null) {
                                nodeHealthState = new RT2NodeHealthState(nodeUuid.toString(), "Unknown", RT2NodeHealth.RT2_NODE_HEALTH_NOT_MEMBER_ANYMORE, RT2NodeHealth.RT2_NODE_TYPE_LITE_MEMBER, thisMemberUuidString);
                                nodeHealthMap.set(nodeUuid.toString(), nodeHealthState);
                                result = true;
                            } else if (StringUtils.isEmpty(cleanupUUID) || !isActiveHzMember(cleanupUUID)) {
                                nodeHealthState.setCleanupUUID(thisMemberUuidString);
                                result = true;
                            }
                        }
                    }
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    log.warn("RT2GarbageCollector caught InterruptedException while trying to handle document entries in DocOnNodeMap with unknown member uuid " + nodeUuid.toString(), e);
                } catch (Exception e) {
                    log.warn("RT2GarbageCollector caught exception  while trying to handle document entries in DocOnNodeMap with unknown member uuid " + nodeUuid.toString(), e);
                } finally {
                    if (locked)
                        hzNodeHealthMap.unlock(nodeUuid.getValue().toString());
                }
            }
        } catch (Exception e) {
            log.warn("Trying analyze health map entry or try to lock for node " + nodeUuid.getValue().toString() + " failed with exception", e);
        }

        return result;
    }

    //-------------------------------------------------------------------------
    private boolean shouldThisNodeTryToTakeOverOwnershipForNotMemberNode(RT2NodeUuidType notMemberUuid) {
        boolean tryToTakeOver = false;

        try {
            final LocalDateTime now = LocalDateTime.now();
            final LocalDateTime lastTime = docOnNodeEntriesToGC.get(notMemberUuid);

            if (lastTime != null) {
                if ((lastTime.plusSeconds(minWaitTimeBeforeGcAtomicLong).isBefore(now))) {
                    docOnNodeEntriesToGC.remove(notMemberUuid);
                    tryToTakeOver = true;
                }
            } else {
                final RT2NodeHealthState nodeHealthState = nodeHealthMap.get(notMemberUuid.toString());
                if (nodeHealthState != null) {
                    final String nodeState = nodeHealthState.getState();
                    final String cleanupUUID = nodeHealthState.getCleanupUUID();
                    if ((RT2NodeHealth.RT2_NODE_HEALTH_SHUTTING_DOWN.equals(nodeState)) ||
                        StringUtils.isEmpty(cleanupUUID) || !isActiveHzMember(cleanupUUID)) {
                        docOnNodeEntriesToGC.put(notMemberUuid, now);
                        log.debug("RT2GarbageCollector marked node {} for gc as node is not member and health state does not indicate that a clean-up process is running", notMemberUuid);
                    }
                } else {
                    docOnNodeEntriesToGC.put(notMemberUuid, now);
                    log.debug("RT2GarbageCollector marked node {} for gc as node is not member and there is no health state available", notMemberUuid);
                }
            }
        } catch (Exception e) {
            log.warn("Trying analyze health map entry for clean-up or try to lock for node " + notMemberUuid.toString() + " failed with exception", e);
        }

        return tryToTakeOver;
    }

    //-------------------------------------------------------------------------
    private boolean isActiveHzMember(String nodeUuid) {
        if (StringUtils.isEmpty(nodeUuid))
            return false;

        boolean isActive = false;
        final Set<Member> members = hazelcastInstance.getCluster().getMembers();
        if (members != null) {
            isActive = members.stream().anyMatch(m -> nodeUuid.equals(m.getUuid()));
        }
        return isActive;
    }

    //-------------------------------------------------------------------------
    private void removeDocOnNodeEntryAndAtomicLongForDocUid(RT2DocUidType docUid, boolean destroyRefCountGlobally) {
        RT2DocOnNodeMap aDocOnNodeMap = docOnNodeMap;
        log.info("Removing docOnNodeMap entry with key {}", docUid.getValue());
        ClusterLock clusterLock = clusterLockService.getLock(docUid);
        try {
            clusterLock.lock();
            if (aDocOnNodeMap.remove(docUid.getValue()) == null) {
                LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, docUid.getValue());
                log.error("Cannot remove entry with doc-uid {} from DocOnNodeMap", docUid.getValue());
            }

            RT2DocInfo aDocInfo = docInfoRegistry.getDocInfoWithoutCreate(docUid);
            if (destroyRefCountGlobally && (aDocInfo == null)) {
                // Attention: This is a workaround to destroy the ref count four line later. The RT2DistributedDocInfo
                // just contains all known document instance that were created on this node. In case of a global
                // clean-up this node never has an entry, but we need it to destroy it. As we could see in dumps
                // of other clusters in most cases a AtomicLong instance is present when we need to clean-up the
                // DocOnNodeMap for a stale groupware node.
                aDocInfo = new RT2DocInfo(hazelcastInstance, docUid.toString());
            }
            if (aDocInfo != null) {
                aDocInfo.destroyRefCount4Clients();
                docOnNodeMapEntryRemovedCounter.inc();
            } else {
                LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, docUid.getValue());
                log.warn("No RefCount4Clients for document with doc-uid {} found!", docUid.getValue());
            }
        } catch (ClusterLockException e) {
            // Ignore, try again next time
        } finally {
            clusterLock.unlock();
            LogProperties.removeProperty(LogProperties.Name.RT2_DOC_UID);
        }
    }

    //-------------------------------------------------------------------------
    private void removeCleanedUpMemberFromHealthMap(final RT2NodeUuidType nodeUuToCleanup) throws OXException {
        log.info("Removing unknown groupware member {} from NodeHealthMap", nodeUuToCleanup.toString());
        nodeHealthMap.remove(nodeUuToCleanup.toString());
    }

    //-------------------------------------------------------------------------
    private /* no synchronized */ Set<RT2DocProcessor> retrieveDocProcessorsWithoutClients (final Set<WeakReference<RT2DocProcessor>> aLocalDocProcessors)
    {
        return aLocalDocProcessors.stream()
                                  .map(WeakRefUtils::getHardRef)
                                  .filter(Objects::nonNull)
                                  .filter(docProc -> CollectionUtils.isEmpty(docProc.getClientsInfo()))
                                  .collect(Collectors.toSet());
    }

    private class FinalRt2DocProcessorRemoveThread implements Runnable {

    	private final Set<RT2DocUidType> possibleDocProcessorToRemove;

    	public FinalRt2DocProcessorRemoveThread(Set<RT2DocUidType> possibleDocProcessorToRemove) {
    		this.possibleDocProcessorToRemove = possibleDocProcessorToRemove;
    	}

		@Override
		public void run() {
			final Set<WeakReference<RT2DocProcessor>> aLocalDocProcessors = docProcMngr.getWeakReferenceToDocProcessors();
			final Set<RT2DocProcessor> aDocProcessorsWOClients = retrieveDocProcessorsWithoutClients(aLocalDocProcessors);
			final RT2DocInfoRegistry       aDocRegistry        = docInfoRegistry;
			for (RT2DocProcessor docProc : aDocProcessorsWOClients) {
				if (possibleDocProcessorToRemove.contains(docProc.getDocUID())) {
	                final RT2DocUidType     sDocUID   = docProc.getDocUID();
	                final RT2DocInfo aDocInfo  = aDocRegistry.peekDocInfo(sDocUID);
					destroyRefCount4Clients(sDocUID, aDocInfo);
					docProcMngr.docProcessorDisposed(docProc);
					LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, sDocUID.getValue());
					log.info("Removed doc processor with docUid {} because there is no client for the last 5 minutes.", sDocUID);
					LogProperties.remove(LogProperties.Name.RT2_DOC_UID);
					docProcRemovedCounter.inc();
				}
			}
		}
    }
}
