package com.openexchange.office.rt2.core;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.hazelcast.core.IAtomicLong;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.cache.RT2DocInfo;
import com.openexchange.office.rt2.core.cache.RT2HazelcastHelperService;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorManager;
import com.openexchange.office.rt2.core.jms.RT2JmsMessageSender;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyRegistry;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.AdminMessage;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.AdminMessageType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ClientInfo;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ClientUidType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.DocUidType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.MessageTimeType;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.ServerIdType;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.tools.annotation.Async;

@Service
@Async(initialDelay=5, period=5, timeUnit=TimeUnit.MINUTES)
public class RT2DocProcessorClientExistsTester implements Runnable {
	
	private static final Logger log = LoggerFactory.getLogger(RT2DocProcessorClientExistsTester.class);
	
	//---------------------------------Services----------------------------------------------------
	@Autowired
	private RT2JmsMessageSender jmsMessageSender;
	
	@Autowired
	private RT2DocProxyRegistry docProxyRegistry;
	
	@Autowired
	private RT2DocProcessorManager docProcMngr;
	
	@Autowired
	private RT2HazelcastHelperService hzHelperService;

	//---------------------------------------------------------------------------------------------
	private final AtomicLong currentlyRunning = new AtomicLong(0);	
	
	private Map<UUID, Collection<ClientInfo>> collectedClients = new HashMap<>();
	
	private Collection<String> markedAtomicsForDelete = new HashSet<>();
	
	private Pattern docUidOfAtomicLongNamePattern = Pattern.compile("rt2cache.atomic.(.*).refcount.clients.a");
	
	@Override
	public void run() {
		if (currentlyRunning.compareAndSet(0l, System.currentTimeMillis())) {
			String nodeUuid = hzHelperService.getHazelcastLocalNodeUuid();
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2DocProcessorClientExistsTester");
            LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, nodeUuid);        				
			collectedClients.clear();
			if (hzHelperService.getCountMembers() == 1) {
				processUpdateCurrClientListOfDocProcResponse(UUID.fromString(nodeUuid), System.currentTimeMillis(), new HashSet<>());
			} else {
				AdminMessage adminRequest = AdminMessage.newBuilder()
														.setMsgType(AdminMessageType.UPDATE_CURR_CLIENT_LIST_OF_DOC_PROCESSOR_REQUEST)
														.setMessageTime(MessageTimeType.newBuilder().setValue(System.currentTimeMillis()))
														.setServerId(ServerIdType.newBuilder().setValue(nodeUuid))
														.setOriginator(ServerIdType.newBuilder().setValue(nodeUuid))
														.build();
				jmsMessageSender.sendAdminMessage(adminRequest);
			}
		} else {
			LocalDateTime started =
				    LocalDateTime.ofInstant(Instant.ofEpochMilli(currentlyRunning.get()), ZoneId.systemDefault());
			LocalDateTime now = LocalDateTime.now();
			Duration duration = Duration.between(now, started);
			if (duration.toMinutes() > 10) {				
				currentlyRunning.set(0l);
				run();
			}
		}
	}
	
	public void processUpdateCurrClientListOfDocProcResponse(UUID nodeUuid, long msgDate, Collection<ClientInfo> clientUids) {
		if (msgDate < currentlyRunning.get()) {
			return;
		}
		if (currentlyRunning.get() > 0l) {
			if (collectedClients.containsKey(nodeUuid)) {
				currentlyRunning.set(0l);
				return;
			}
			if (!clientUids.isEmpty()) {
				collectedClients.put(nodeUuid, clientUids);
			}
			if (collectedClients.size() == hzHelperService.getCountMembers() - 1) {
				collectedClients.put(UUID.fromString(hzHelperService.getHazelcastLocalNodeUuid()), 
						docProxyRegistry.listAllDocProxies().stream().map(p -> 
						 	ClientInfo.newBuilder()
						 		.setClientUid(ClientUidType.newBuilder().setValue(p.getClientUID().getValue()))
						 		.setDocUid(DocUidType.newBuilder().setValue(p.getDocUID().getValue()))
						 	.build()).collect(Collectors.toSet()));						
				Set<RT2CliendUidType> res = new HashSet<>();
				collectedClients.values().forEach(col -> {
					col.forEach(c -> res.add(new RT2CliendUidType(c.getClientUid().getValue())));
				});
				docProcMngr.getDocProcessors().forEach(r -> r.updateClientsInfo(res));
				checkAtomicLongs();
			}
			currentlyRunning.set(0l);
		}
	}	
	
	private void checkAtomicLongs() {
		Map<String, IAtomicLong> allAtomicLongs = hzHelperService.getAllClientRefCountObjects();
		Set<String> isDocProxies = new HashSet<>();
		collectedClients.values().forEach(col -> col.forEach(c -> isDocProxies.add(RT2DocInfo.generateNameFromDocUid(c.getDocUid().getValue()))));
		Collection<String> newMarkedAtomicForDelete = new HashSet<>();
		for (Map.Entry<String, IAtomicLong> e : allAtomicLongs.entrySet()) {
			if (!isDocProxies.contains(e.getKey())) {
				if (markedAtomicsForDelete.contains(e.getKey())) {
					Matcher m = docUidOfAtomicLongNamePattern.matcher(e.getKey());				
					if (m.find()) {
						LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, m.group(1));
						log.info("Removing AtomicLong with name {}", e.getKey());
					}								
					e.getValue().destroy();
					LogProperties.remove(LogProperties.Name.RT2_DOC_UID);
				} else {
					newMarkedAtomicForDelete.add(e.getKey());
				}
			}
		}
		markedAtomicsForDelete = newMarkedAtomicForDelete;
		if (!markedAtomicsForDelete.isEmpty()) {
			for (String str : markedAtomicsForDelete) {
				Matcher m = docUidOfAtomicLongNamePattern.matcher(str);				
				if (m.find()) {
					LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, m.group(1));
					log.info("Marked AtomicLong for remove with name {}, current atomic longs: {}, currentDocProxys: {}", str, allAtomicLongs.keySet(), collectedClients.values());
				}				
				LogProperties.remove(LogProperties.Name.RT2_DOC_UID);
			}
		}
	}

	public Collection<String> getMarkedAtomicsForDelete() {
		return new HashSet<>(markedAtomicsForDelete);
	}
	
	public long getCurrentlyRunning() {
		return currentlyRunning.get();
	}
}
