
package com.openexchange.office.rest.mention;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.UserizedFile;
import com.openexchange.office.rest.DocumentRESTAction;
import com.openexchange.office.rest.tools.RequestDataHelper;
import com.openexchange.office.tools.common.error.HttpStatusCode;
import com.openexchange.office.tools.service.files.FileHelperService;
import com.openexchange.office.tools.service.files.FileHelperServiceFactory;
import com.openexchange.office.tools.service.files.FolderHelperService;
import com.openexchange.office.tools.service.files.FolderHelperServiceFactory;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.user.User;
import com.openexchange.user.UserService;

@RestController
public class GetDocumentUserWriteAccess extends DocumentRESTAction<AJAXRequestResult> {

    private static final Logger log = LoggerFactory.getLogger(GetDocumentUserWriteAccess.class);

    private ObjectMapper objMapper = new ObjectMapper();

    @Autowired
    private FileHelperServiceFactory fileHelperServiceFactory;

    @Autowired
    private FolderHelperServiceFactory folderHelperServiceFactory;

    @Autowired
    private UserService userService;

    @Override
    public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {
        String requestBodyData = requestData.getParameter(RequestDataHelper.KEY_REQUESTDATA);

        DocumentUserWriteAccessRequestData requestDataObject;

        try {
            requestDataObject = objMapper.readValue(requestBodyData, DocumentUserWriteAccessRequestData.class);
        } catch (JsonParseException | JsonMappingException e) {
            log.info(e.getMessage());
            return getAJAXRequestResultWithStatusCode(HttpStatusCode.BAD_REQUEST.getStatusCode());
        } catch (IOException e) {
            log.info(e.getMessage());
            return getAJAXRequestResultWithStatusCode(HttpStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());
        }

        final FileHelperService fileHelper = fileHelperServiceFactory.create(session);
        final FolderHelperService folderHelper = folderHelperServiceFactory.create(session);

        Set<Integer> userIdsWithoutWriteAccess = new HashSet<>();
        Set<String> userEmailsWithoutWriteAccess = new HashSet<>();

        // Test if the session user can write to file
        if (fileHelper.canReadFile(requestDataObject.getFolderId(), requestDataObject.getFileId(), session.getUserId())) {

            // All users to test if the users can write to document
            List<Integer> userIds = new ArrayList<Integer>();
            userIds.addAll(requestDataObject.getUserIds());

            // Map of email to userId and add userIds to the userId list to test the write access
            Map<String, Integer> userEmails = new HashMap<>();
            for (String email : requestDataObject.getEmails()) {
                try {
                    User user = userService.searchUser(email, session.getContext(), true, true, false);
                    userEmails.put(email, user.getId());
                    userIds.add(user.getId());
                } catch (Exception e) {
                    // If the email is not known a exception occurred
                    userEmails.put(email, null);
                    // do nothing
                }
            }

            Set<Integer> userIdsWithWriteAccess = new HashSet<>();
            // test the folder write access of the users
            userIdsWithWriteAccess.addAll(folderHelper.canWriteToFolder(requestDataObject.getFolderId(), session.getContext(), userIds));
            // test the file write access of the users
            userIdsWithWriteAccess.addAll(fileHelper.canWriteToFile(requestDataObject.getFileId(), session.getContext(), userIds));

            /*
             * Workaround
             *
             * If it's a shared document it is currently not possible with the session of the shared document user to
             * get the permissions for the original document. I normal case the creator of the document works on the original
             * document and not on the shared document so we can't get the permissions for the creator.
             * In this case we assume that the creator has write permissions and add the user to the list of users with write access.
             *
             * DOCS-2193
             */
            if (userIdsWithWriteAccess.size() != userIds.size()) {
                File file = fileHelper.getFileMetaData(requestDataObject.getFileId());
                // Test if the document creator ID is in the request userIds list and if it has no write access
                if (userIds.contains(file.getCreatedBy()) && !userIdsWithWriteAccess.contains(file.getCreatedBy())) {
                    if (UserizedFile.class.isInstance(file)) {
                        String originalId = ((UserizedFile) file).getOriginalId();
                        // Test if it is a shared file
                        if (originalId != null && !originalId.equals(requestDataObject.getFileId())) {
                            userIdsWithWriteAccess.add(file.getCreatedBy());
                        }
                    }
                }
            }

            // filter user Ids without write access
            for (int userId : requestDataObject.getUserIds()) {
                if (!userIdsWithWriteAccess.contains(userId) && requestDataObject.getUserIds().contains(userId)) {
                    userIdsWithoutWriteAccess.add(userId);
                }
            }

            // filter user emails without write access
            for (Entry<String, Integer> userEmail : userEmails.entrySet()) {
                if (userEmail.getValue() == null || !userIdsWithWriteAccess.contains(userEmail.getValue())) {
                    userEmailsWithoutWriteAccess.add(userEmail.getKey());
                }
            }
        }

        try {
            // send the users Ids/emails without write access
            return getAJAXRequestResultWithUsersWithoutWriteAccess(userIdsWithoutWriteAccess, userEmailsWithoutWriteAccess);
        } catch (JSONException e) {
            log.info(e.getMessage());
            return getAJAXRequestResultWithStatusCode(HttpStatusCode.INTERNAL_SERVER_ERROR.getStatusCode());
        }
    }


    private AJAXRequestResult getAJAXRequestResultWithStatusCode(int statusCode) {
        final AJAXRequestResult requestResult = new AJAXRequestResult();
        requestResult.setHttpStatusCode(statusCode);

        return requestResult;
    }

    private AJAXRequestResult getAJAXRequestResultWithUsersWithoutWriteAccess(Set<Integer> userIdsWithoutWriteAccess, Set<String> userEmailsWithoutWriteAccess) throws JSONException {
        JSONObject resultObject = new JSONObject();

        if (!userIdsWithoutWriteAccess.isEmpty()) {
            resultObject.put("userIds", new JSONArray(userIdsWithoutWriteAccess));
        }

        if (!userEmailsWithoutWriteAccess.isEmpty()) {
            resultObject.put("emails", new JSONArray(userEmailsWithoutWriteAccess));
        }

        return new AJAXRequestResult(resultObject);
    }

}
