/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFileStorageFolder;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.office.tools.common.EncodeUtility;
import com.openexchange.office.tools.common.IOHelper;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.service.files.FileHelperService;
import com.openexchange.office.tools.service.files.FileHelperServiceFactory;
import com.openexchange.office.tools.service.files.FolderHelperService;
import com.openexchange.office.tools.service.files.FolderHelperServiceFactory;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.tools.session.ServerSession;

@RestController
public class LogClientDataAction extends DocumentRESTAction {

	private static final Logger LOG = LoggerFactory.getLogger(LogClientDataAction.class);
	public static final String CRASH_DATA_FOLDER_NAME = "crashdata";

	@Autowired
	private FileHelperServiceFactory fileHelperServiceFactory;
	
	@Autowired
	private FolderHelperServiceFactory folderHelperServiceFactory;

	@Override
	public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) throws OXException {
        if ((null == session) || (session.isAnonymous())) {
            return getAJAXRequestResultWithStatusCode(401);
        }

        final boolean logAnonymous = Boolean.valueOf(request.getParameter("logAnonymous"));
        
        final String logData = request.getParameter("logData");

        final String folderId = request.getParameter("folderId");
        final String fileId = request.getParameter("fileId");

        final String docUid = request.getParameter("docUid");
        final String fileName = request.getParameter("fileName");

        final String errorCode = request.getParameter("errorCode");
        final String userId = request.getParameter("userId");

        AJAXRequestResult res = getAJAXRequestResultWithStatusCode(201);
    	if (StringUtils.isNotBlank(logData)) {

	        String crashDataFolderId = getCrashdataFolder(session);

	        InputStream inputStream = null;
	        final String mimeType = "text/plain";
	        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd_HHmm");

	        // we should never use the fileId/folderId directly to create a file name - it must be
	        // encoded, because we don't know the format and there are several characters which are
	        // forbidden in a filename
	        String crashDataFilename = EncodeUtility.encodeForFileName(fileName) + EncodeUtility.encodeForFileName(fileId) + formatter.format(LocalDateTime.now());
	        final File file = getNewFile(request, crashDataFolderId, crashDataFilename, mimeType, "txt", null, false);
	        try {
	        	inputStream = IOUtils.toInputStream(logData, Charset.defaultCharset());

            	final FileHelperService fileHelperService = fileHelperServiceFactory.create(session);
	        	
	        	ErrorCode errorCodeTmp = fileHelperService.createFileAndWriteStream(file, inputStream);
	        	if (errorCodeTmp.isError()) {
	        		throw AjaxExceptionCodes.UNEXPECTED_ERROR.create(errorCodeTmp.getDescription());
	        	}
	        } finally {
	        	IOHelper.closeQuietly(inputStream);
			}
        } else {
	    	if (logAnonymous) {
	    		LOG.error("DocumentError written for user {} for file with docUid {} ({})", userId, docUid, errorCode);
	       	} else {
	       		LOG.error("DocumentError written for user {} for file files&folder={}&id={} ({})", userId, folderId, fileId, errorCode);
	        }
        }
        return res;
	}

	private String getCrashdataFolder(ServerSession session) throws OXException {
		final FolderHelperService folderHelperService = folderHelperServiceFactory.create(session);
		final String userFolderId = folderHelperService.getUserFolderId(session);
		final FileStorageFolder [] subFolders = folderHelperService.getSubfolders(userFolderId, true);

		for (FileStorageFolder subFolder : subFolders) {
			if (subFolder.getName().equals(CRASH_DATA_FOLDER_NAME)) {
				return subFolder.getId();
			}
		}

		// Not found!
		DefaultFileStorageFolder fileStorageFolder = new DefaultFileStorageFolder();
		fileStorageFolder.setName(CRASH_DATA_FOLDER_NAME);
		fileStorageFolder.setParentId(userFolderId);
		folderHelperService.createFolder(fileStorageFolder);
		return getCrashdataFolder(session);
	}

    /**
     * Provide a AJAXRequestResult with a set status code.
     *
     * @param statusCode the status code to be set at the resulting AJAXRequestResult instance
     * @return the AJAXRequestResult with the provided status code set
     * @throws OXException
     */
    private AJAXRequestResult getAJAXRequestResultWithStatusCode(int statusCode) throws OXException {

        final AJAXRequestResult requestResult = new AJAXRequestResult();
        requestResult.setHttpStatusCode(statusCode);
        return requestResult;
    }
}
