/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ot.spreadsheet;

import com.openexchange.office.filter.core.spreadsheet.Interval;
import com.openexchange.office.tools.doc.DocumentFormat;

/**
 * An instance of this class contains the maximum possible column index and row
 * index of cell addresses in a spreadsheet document, and provides methods to
 * deal with index intervals (columns and rows), cell addresses, and cell range
 * addresses.
 *
 * @param colCount
 *  The total number of columns contained in a sheet.
 *
 * @param rowCount
 *  The total number of rows contained in a sheet.
 */
public class AddressFactory {

    final int maxColumns;

    final int maxRows;

    public AddressFactory(DocumentFormat format) {
        if(format == DocumentFormat.ODS) {
            this.maxColumns = 1024;
            this.maxRows = 65536;
        }
        else {
            this.maxColumns = 16384;
            this.maxRows = 1048576;
        }
    }

    public AddressFactory(int maxColumns, int maxRows) {
        this.maxColumns = maxColumns;
        this.maxRows = maxRows;
    }

    public final int getColCount() {
        return maxColumns;
    }

    public final int getRowCount() {
        return maxRows;
    }

    public int getMaxCol() {
        return maxColumns - 1;
    }

    public int getMaxRow() {
        return maxRows - 1;
    }

    /**
     * Returns the maximum column or row index in a sheet.
     *
     * @param columns
     *  Whether to return the maximum column index (`true`), or the maximum row
     *  index (`false`).
     *
     * @returns
     *  The maximum column or row index in a sheet.
     */
    public int getMaxIndex(boolean columns) {
        return columns ? getMaxCol() : getMaxRow();
    }

    /**
     * Returns the index interval covering all columns or rows in a sheet.
     *
     * @param columns
     *  Whether to return the column interval (`true`), or the row interval
     *  (`false`).
     *
     * @returns
     *  The full column or row interval in a sheet. Each invocation of this
     *  method creates a new interval object that can be modified further.
     */
    public Interval getFullInterval(boolean columns) {
        return Interval.createInterval(columns, 0, getMaxIndex(columns));
    }
}
