/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.imagemgr.impl;

import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.openexchange.office.tools.annotation.Async;
import com.openexchange.osgi.ExceptionUtils;

@Service
@Async(initialDelay=ResourceManagers.CHECK_UNAVAIL_REFS_FREQUENCY, period=ResourceManagers.CHECK_UNAVAIL_REFS_FREQUENCY, timeUnit=TimeUnit.MILLISECONDS)
public class ResourceManagers implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ResourceManagers.class);

	protected static final long CHECK_UNAVAIL_REFS_FREQUENCY = 30000;

	private Map<String, WeakReference<ResourceManagerImpl>> createdResMgrs = new ConcurrentHashMap<>();

	public void register(ResourceManagerImpl resourceManager) {
		final WeakReference<ResourceManagerImpl> weakRef = new WeakReference<>(resourceManager);
		createdResMgrs.put(resourceManager.getUUID(), weakRef);
	}

	public Set<ResourceManagerImpl> getRegisteredResourceManagers() {
		return createdResMgrs.keySet().stream().map(k -> {
			final WeakReference<ResourceManagerImpl> weakRef = createdResMgrs.get(k);
			return weakRef.get();
		}).filter(r -> r != null).collect(Collectors.toSet());
	}

	@Override
	public void run() {
		try {
			createdResMgrs.entrySet().removeIf(e -> {
				return isNotValidInstance(e.getValue());
			});
		} catch (Throwable t) {
			ExceptionUtils.handleThrowable(t);
			LOG.warn("Exception caught trying to clean-up registered ResourceManagers", t);
		}
	}

	private boolean isNotValidInstance(WeakReference<ResourceManagerImpl> weakRef) {
		final ResourceManagerImpl instance = (weakRef != null) ? weakRef.get() : null;
		return (instance == null) ? true : instance.isDisposed();
	}
}
