/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.imagemgr;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.common.IOHelper;

@Service
@RegisteredService
public class ResourceFactory {
	
	private static final Logger log = LoggerFactory.getLogger(ResourceFactory.class);
	
	@Autowired
	private ManagedFileManagement fileMgr;
	
	public Resource create(byte[] buffer) {
		String sha256 = Resource.getSHA256(buffer);
		return create(sha256, buffer);
	}
	
	public Resource create(String sha256, byte[] buffer) {
		String uid = null;
		ManagedFile managedFile = null;
	    if (null != buffer) {
	    	uid = Resource.getUIDFromData(sha256, buffer);

	    	 // first try to find out, if we already have the managed file
	    	try {
	    		final String managedId = Resource.getManagedIdFromUID(uid);
	    		if (fileMgr.contains(managedId))
	    			managedFile = fileMgr.getByID(managedId);
	    	} catch (OXException e) {
              // nothing to do - exception means managed file is not known
	    	}

	    	if (null == managedFile) {
	    		// create a new managed file
	    		try {
	    			final InputStream inputStream = new ByteArrayInputStream(buffer);
	    			managedFile = fileMgr.createManagedFile(Resource.getManagedIdFromUID(uid), inputStream, true);
	    			IOHelper.closeQuietly(inputStream);
	    		} catch (OXException e) {
	    			log.error(e.getMessage(), e);
	    		}
	    	}
	    }
	    return new Resource(managedFile, uid);
	}
	
	public Resource create(String id) throws OXException {
		String uid = Resource.getUIDFromManagedId(id);
		if ((StringUtils.isNotEmpty(uid) && (fileMgr.contains(id)))) {
			ManagedFile managedFile = fileMgr.getByID(id);
			return new Resource(managedFile, uid);
        }
		return null;
    }
}	   
