/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.imagemgr;

import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.office.tools.common.IOHelper;

/**
 * {@link Resource}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class Resource {

    public static final String RESOURCE_UID_PREFIX = "xid";
    private static final Logger LOG = LoggerFactory.getLogger(Resource.class);
    private static final String idPrefix = "office.resource0x";

    private ManagedFile m_managedFile = null;
    private String m_uid = null;

    public Resource(ManagedFile managedFile, String uid) {
    	this.m_managedFile = managedFile;
    	this.m_uid = uid;
    }
    
    @Override
    public String toString() {
        final StringBuilder tmp = new StringBuilder("Resource: ");
        tmp.append("ManagedId: ").append(getManagedId());
        if (m_managedFile != null) {
            tmp.append(", managed file: ").append(m_managedFile.getFile());
            tmp.append(", last access: ").append(convertToLocalDateTime(m_managedFile.getLastAccess()));
        }
        return tmp.toString();
    }

    /**
     * @return The content of this resource as a byte array
     */
    public byte[] getBuffer() {
        byte[] ret = null;

        if (null != m_managedFile) {
            try {
                final InputStream inputStream = m_managedFile.getInputStream();

                if (null != inputStream) {
                    try {
                        ret = IOUtils.toByteArray(inputStream);
                    } catch (IOException e) {
                        LOG.error(e.getMessage(), e);
                    } finally {
                        IOHelper.closeQuietly(inputStream);
                    }
                }
            } catch (OXException e) {
                LOG.error(e.getMessage());
            }
        }

        return ret;
    }

    /**
     * @return The UID of the contained data
     */
    public String getUID() {
        return m_uid;
    }

    /**
     * @return The id of the distributed managed file or null if the id is not valid
     */
    public String getManagedId() {
        return (null != m_uid) ? getManagedIdFromUID(getUID()) : null;
    }

    /**
     * @return The resource identifier of this resource as a string according to the following schema:
     * Resource.RESOURCE_UID_PREFIX + HexValueString
     */
    public String getResourceName() {
        return getResourceNameFromUID(m_uid);
    }

    /**
     *
     */
    public void touch() {
        if (null != m_managedFile) {
            m_managedFile.touch();
        }
    }

    /**
     *
     */
    public void delete() {
        if (null != m_managedFile) {
            m_managedFile.delete();
        }
    }

    /**
     * @param data the data where the SHA-256 hash should be calculated from
     * @return the SHA-256 hash of the provided data or null if not possible
     */
    public static String getSHA256(byte[] data) {
        String result = null;
        try {
            result = Hex.encodeHexString(MessageDigest.getInstance("SHA-256").digest(data), true);
        } catch (NoSuchAlgorithmException e) {
            LOG.error("Not possible to determine SHA-256 hash from data", e);
        }
        return result;
    }

    /**
     * @param hash of the managed file (is the SHA256 hash of the resource data)
     * @return
     */
    public static String getManagedIdFromUID(String hash) {
        return (idPrefix + hash);
    }

    /**
     * @param managedFileId the managed file id
     * @return
     */
    public static String getUIDFromManagedId(String managedFileId) {
        String uid = null;

        if (StringUtils.isNotEmpty(managedFileId) && managedFileId.startsWith(idPrefix)) {
            uid = managedFileId.substring(idPrefix.length());
        }

        return uid;
    }

    /**
     * @param uid The hash of a resource
     * @return The resource identifier for the given uid as a string according to the
     * following schema: RESOURCE_UID_PREFIX + HexValueString, or null in case of an invalid uid
     */
    public static String getResourceNameFromUID(String hash) {
        return ((null != hash) ? (RESOURCE_UID_PREFIX + hash) : null);
    }

    /**
     * @param resourceName A resource name with the following schema: Resource.RESOURCE_UID_PREFIX + HexValueString
     * @return The hash of a resource name, or 0 in case of an invalid resource name
     */
    public static String getUIDFromResourceName(String resourceName) {
        return ((null != resourceName) ? resourceName.substring(3) : null);
    }

    /**
     * @param sha256
     * @param resdata
     * @return
     * @throws NoSuchAlgorithmException 
     */
    public static String getUIDFromData(String sha256, byte[] resdata) {
        return sha256;
    }

    /**
     * 
     * @param millisInEpoch
     * @return
     */
    private String convertToLocalDateTime(long millisInEpoch) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(millisInEpoch), ZoneOffset.UTC).toString();
    }

}
