/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.docx;

import java.io.InputStream;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.docx4j.jaxb.Context;
import org.json.JSONObject;
import org.springframework.stereotype.Service;

import com.openexchange.office.filter.api.DocumentProperties;
import com.openexchange.office.filter.api.IImporter;
import com.openexchange.office.filter.api.IPartImporter;
import com.openexchange.office.filter.ooxml.OfficeOpenXMLOperationDocument;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.common.osgi.context.OsgiBundleContextAndActivator;
import com.openexchange.office.tools.common.osgi.context.OsgiBundleContextAware;
import com.openexchange.session.Session;

/**
 * {@link Importer}
 *
 * @author <a href="mailto:sven.jacobi@open-xchange.com">Sven Jacobi</a>
 */
@Service
@RegisteredService
public class Importer implements IImporter, IPartImporter, OsgiBundleContextAware {

	private OsgiBundleContextAndActivator bundleCtx;
	
    /**
     * Initializes a new {@link Importer}.
     */
    public Importer() {
        Context.getWmlObjectFactory();
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.office.IImporter#createOperations(java.io.InputStream)
     */
    @Override
    public JSONObject createOperations(Session session, InputStream inputDocumentStm, DocumentProperties documentProperties) {
        boolean debugGetDefaultDocument = false;
        if(debugGetDefaultDocument) {   // just a simple helper method to store our in memory created
            try(DocxOperationDocument emptyDoc = new DocxOperationDocument(null, null, documentProperties)) {
            	bundleCtx.prepareObject(emptyDoc);
                emptyDoc.createPackage();
                emptyDoc.writeFile("/tmp/default.oxt");    // template documents...
            }
            catch(Throwable e) {
                OfficeOpenXMLOperationDocument.rethrowFilterException(e, null);
            }
        }

        JSONObject operations = null;
        try(DocxOperationDocument operationDocument = new DocxOperationDocument(session, null, documentProperties)) {
        	bundleCtx.prepareObject(operationDocument);
            operationDocument.loadDocument(inputDocumentStm, true);
            operations = operationDocument.getOperations();
        }
        catch(Throwable e) {
            OfficeOpenXMLOperationDocument.rethrowFilterException(e, null);
        }
        return operations;
    }

    @Override
    public InputStream getDefaultDocument(InputStream templateDocument, DocumentProperties documentProperties) {
        InputStream defaultDocument = null;
        try(DocxOperationDocument operationDocument = new DocxOperationDocument(null, null, documentProperties)) {
        	bundleCtx.prepareObject(operationDocument);
            if(templateDocument!=null) {
                operationDocument.loadDocument(templateDocument, false);
            }
            else {
                operationDocument.createPackage();
            }
        	operationDocument.applyDefaultDocumentProperties();
        	defaultDocument = operationDocument.save();
        }
        catch(Throwable e) {
            OfficeOpenXMLOperationDocument.rethrowFilterException(e, null);
        }
        finally {
        	IOUtils.closeQuietly(templateDocument);
        }
        return defaultDocument;
    }

	@Override
	public void initPartitioning(Session session, InputStream inputDocument, DocumentProperties documentProperties) {
	    try(DocxOperationDocument operationDocument = new DocxOperationDocument(session, null, documentProperties)) {
	    	bundleCtx.prepareObject(operationDocument);
	        operationDocument.loadDocument(inputDocument, true);
	        documentProperties.put(DocumentProperties.PROP_DOCUMENT, operationDocument);
	    }
	    catch(Throwable e) {
	        OfficeOpenXMLOperationDocument.rethrowFilterException(e, null);
	    }
	}

	@Override
	public Map<String, Object> getMetaData(DocumentProperties documentProperties) {
	    Map<String, Object> metaData = null;
	    final DocxOperationDocument operationDocument = (DocxOperationDocument)documentProperties.get(DocumentProperties.PROP_DOCUMENT);
	    try {
	        operationDocument.registerMemoryListener();
	        metaData = operationDocument.getMetaData();
	    }
	    catch(Throwable e) {
	        OfficeOpenXMLOperationDocument.rethrowFilterException(e, operationDocument.getPackage());
	    }
	    finally {
	        operationDocument.close();
	    }
	    return metaData;
	}

	@Override
	public Map<String, Object> getActivePart(DocumentProperties documentProperties) {
	    Map<String, Object> metaData = null;
        final DocxOperationDocument operationDocument = (DocxOperationDocument)documentProperties.get(DocumentProperties.PROP_DOCUMENT);
	    try {
	        operationDocument.registerMemoryListener();
	        metaData = operationDocument.getActivePart();
	    }
	    catch(Throwable e) {
	        OfficeOpenXMLOperationDocument.rethrowFilterException(e, operationDocument.getPackage());
	    }
	    finally {
	        operationDocument.close();
	    }
	    return metaData;
	}

	@Override
	public Map<String, Object> getNextPart(DocumentProperties documentProperties) {
	    Map<String, Object> metaData = null;
	    final DocxOperationDocument operationDocument = (DocxOperationDocument)documentProperties.get(DocumentProperties.PROP_DOCUMENT);
	    try {
	        operationDocument.registerMemoryListener();
	        metaData = operationDocument.getNextPart();
	    }
	    catch(Throwable e) {
	        OfficeOpenXMLOperationDocument.rethrowFilterException(e, operationDocument.getPackage());
	    }
        finally {
            operationDocument.close();
        }
	    return metaData;
	}

	@Override
	public void setApplicationContext(OsgiBundleContextAndActivator bundleCtx) {
		this.bundleCtx = bundleCtx;
	}
}
