package com.openexchange.office.documents.access.impl;

import java.io.InputStream;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.springframework.jmx.export.annotation.ManagedResource;
import org.springframework.stereotype.Service;

import com.openexchange.office.tools.annotation.Async;
import com.udojava.jmx.wrapper.JMXBean;
import com.udojava.jmx.wrapper.JMXBeanAttribute;

@Service
@Async(initialDelay=1, period=3, timeUnit=TimeUnit.MINUTES)
@JMXBean
@ManagedResource(objectName="com.openexchange.office:name=CorrelationIdDisposer")
public class CorrelationIdDisposer implements Runnable {

	private ConcurrentHashMap<String, CompletableFuture<InputStream>> correlationIds = new ConcurrentHashMap<>();

	@Override
	public void run() {
		Set<String> toRemove = new HashSet<>();
		correlationIds.forEach(3, (k, v) -> {
			if (v.isDone() && (v.getNumberOfDependents() == 0)) {
				toRemove.add(k);
			}
		});
		toRemove.forEach(s -> correlationIds.remove(s));
	}

	public ImmutablePair<String, CompletableFuture<InputStream>> createCorrelationId() {
		
		CompletableFuture<InputStream> completableFuture = new CompletableFuture<>();
		
		String correlationId = UUID.randomUUID().toString();
        correlationIds.put(correlationId, completableFuture);
        return ImmutablePair.of(correlationId, completableFuture);
	}
	
	public CompletableFuture<InputStream> getCompletableFutureForCorrelationId(String correlationId) {
		CompletableFuture<InputStream> completableFuture = correlationIds.get(correlationId);
		if (completableFuture != null) {
			return completableFuture;
		}
		return null;
	}
	
	public void removeCorrelationId(String correlationId) {
		correlationIds.remove(correlationId);
	}
	
	@JMXBeanAttribute
	public int getCountWaitingCompletableFutures() {
		return correlationIds.size();
	}
}
