/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.dcs.registry;

import static org.apache.commons.lang3.StringUtils.isEmpty;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.database.DatabaseException;
import com.openexchange.office.tools.database.DocumentsDatabase;


/**
 * {@link DCSDatabase}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.1
 */
@Service
@RegisteredService
public class DCSDatabase extends DocumentsDatabase implements InitializingBean {

    private static final Logger log = LoggerFactory.getLogger(DCSDatabase.class);

    @Autowired
    private DcsDatabaseConfiguration connectionData;

    @Override
	public void afterPropertiesSet() throws Exception {
    	log.trace("afterPropertiesSet for DCSDatabase called");
        if (isEmpty(this.connectionData.getConnectionURL()) ) {
            throw new DatabaseException("Property readUrl must not be empty (Example Url: jdbc:mysql://host:port/databasename)");
        }
        setReadConnectionData(connectionData);
        implCreateConnectionPools(true, false); 
	}

    /**
     * @return
     * @throws DatabaseException
     */
    public List<DCSRegistryItem> getRegisteredDCSItems() throws DatabaseException {
        List<DCSRegistryItem> items = new ArrayList<>();

        try (final Connection con = getReadConnection()) {
            final String queryItemsSql = "SELECT ID, Host, Interface, JMSPort, UseSSL FROM DCS";

            try (final PreparedStatement dcsStmt = con.prepareStatement(queryItemsSql);
                 final ResultSet resultSet = dcsStmt.executeQuery()) {

                if (null != resultSet) {
                   while (resultSet.next()) {
                       final DCSRegistryItem newItem = new DCSRegistryItem();
                       int index = 0;

                       newItem.setID(resultSet.getString(++index));
                       newItem.setHost(resultSet.getString(++index));
                       newItem.setInterface(resultSet.getString(++index));
                       newItem.setJMSPort(Integer.valueOf(resultSet.getInt(++index)));
                       newItem.setUseSSL(Boolean.valueOf(resultSet.getBoolean(++index)));

                       items.add(newItem);
                   }
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }
        } catch (Exception e) {
            throw new DatabaseException("Exception caught when querying all registered DCSRegistryItems", e);
        }

        return items;
    }    
}
