/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.datasource.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.openexchange.exception.OXException;
import com.openexchange.mail.compose.Attachment;
import com.openexchange.mail.compose.CompositionSpaceService;
import com.openexchange.mail.compose.CompositionSpaces;
import com.openexchange.office.datasource.access.DataSource;
import com.openexchange.office.datasource.access.DataSourceException;
import com.openexchange.office.datasource.access.MetaData;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.error.ExceptionToErrorCode;
import com.openexchange.office.tools.common.files.FileHelper;
import com.openexchange.office.tools.doc.DocumentFormatHelper;
import com.openexchange.tools.session.ServerSession;


public class DataSourceAccessCompose extends DataSourceAccessBase {
    private static final Logger LOG = LoggerFactory.getLogger(DataSourceAccessCompose.class);

    @Autowired
    private CompositionSpaceService compositionSpaceService;
    
    public DataSourceAccessCompose(ServerSession session) {
        super(session);
    }

    @Override
    public boolean canRead(String folderId, String id, String versionOrAttachmentId) throws DataSourceException {
        if (compositionSpaceService == null)
            throw new DataSourceException(ErrorCode.GENERAL_SERVICE_DOWN_ERROR);

        try {
            final UUID compositionSpaceUuid = CompositionSpaces.parseCompositionSpaceId(id);
            final UUID attachmentUuid = CompositionSpaces.parseAttachmentId(versionOrAttachmentId);
            return (compositionSpaceService.getAttachment(compositionSpaceUuid, attachmentUuid, session) != null);
        } catch (OXException e) {
            LOG.error("DataSourceAccessCompose: Exception caught trying to get attachment from CompositionSpace", e);
            final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_UNKNOWN_ERROR, false);
            throw new DataSourceException(errorCode);
        }
    }

    @Override
    public MetaData getMetaData(String folderId, String id, String versionOrAttachmentId, Optional<String> authCode) throws DataSourceException {
        if (compositionSpaceService == null)
            throw new DataSourceException(ErrorCode.GENERAL_SERVICE_DOWN_ERROR);

        try {
            final UUID compositionSpaceUuid = CompositionSpaces.parseCompositionSpaceId(id);
            final UUID attachmentUuid = CompositionSpaces.parseAttachmentId(versionOrAttachmentId);
            final Attachment attachment = compositionSpaceService.getAttachment(compositionSpaceUuid, attachmentUuid, session);

            final long fileSize = attachment.getSize();
            final String fileName = attachment.getName();
            String mimeType = attachment.getMimeType();

            // try to refine mime type using extension if we see an unknown mime type
            final Map<String, String> docFormatInfo = DocumentFormatHelper.getFormatInfoForMimeTypeOrExtension(mimeType, FileHelper.getExtension(fileName, true));
            if (docFormatInfo != null) {
                mimeType = docFormatInfo.get(DocumentFormatHelper.PROP_MIME_TYPE);
            }

            final String hash = createHashFrom(fileSize, fileName, mimeType);
            return new MetaData(folderId, id, versionOrAttachmentId, fileSize, fileName, hash, mimeType);
        } catch (OXException e) {
            LOG.error("DataSourceAccessCompose: Exception caught trying to get meta data from attachment from CompositionSpace", e);
            final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_UNKNOWN_ERROR, false);
            throw new DataSourceException(errorCode);
        }
    }

    @Override
    public InputStream getContentStream(String folderId, String id, String versionOrAttachmentId, Optional<String> authCode) throws DataSourceException {
        if (compositionSpaceService == null)
            throw new DataSourceException(ErrorCode.GENERAL_SERVICE_DOWN_ERROR);

        try {
            final UUID compositionSpaceUuid = CompositionSpaces.parseCompositionSpaceId(id);
            final UUID attachmentUuid = CompositionSpaces.parseAttachmentId(versionOrAttachmentId);
            final Attachment attachment = compositionSpaceService.getAttachment(compositionSpaceUuid, attachmentUuid, session);
            final long fileSize = attachment.getSize();

            checkFilSizeAndMemoryConsumption(fileSize, null);

            final byte[] data = IOUtils.toByteArray(attachment.getData());
            return new ByteArrayInputStream(data);
        } catch (OXException e) {
            LOG.error("DataSourceAccessCompose: Exception caught trying to get meta data from attachment from CompositionSpace", e);
            final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_UNKNOWN_ERROR, false);
            throw new DataSourceException(errorCode);
        } catch (IOException e) {
            LOG.error("DataSourceAccessCompose: Exception caught trying to get meta data from attachment from CompositionSpace", e);
            throw new DataSourceException(ErrorCode.GENERAL_UNKNOWN_ERROR);
        }
    }

    @Override
    public String getDataSource() {
        return DataSource.COMPOSE;
    }

}
