/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.datasource.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.exception.OXException;
import com.openexchange.groupware.Types;
import com.openexchange.groupware.attach.AttachmentBase;
import com.openexchange.groupware.attach.AttachmentMetadata;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.office.datasource.access.DataSource;
import com.openexchange.office.datasource.access.DataSourceAccess;
import com.openexchange.office.datasource.access.DataSourceException;
import com.openexchange.office.datasource.access.MetaData;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.error.ExceptionToErrorCode;
import com.openexchange.office.tools.common.files.FileHelper;
import com.openexchange.office.tools.doc.DocumentFormatHelper;
import com.openexchange.tools.session.ServerSession;

public class DataSourceAccessCalendarTaskContact extends DataSourceAccessBase implements DataSourceAccess {
    private static final Logger LOG = LoggerFactory.getLogger(DataSourceAccessCalendarTaskContact.class);

	final String dataSource;
	final int moduleId;

	public DataSourceAccessCalendarTaskContact(ServerSession session, String dataSource) throws DataSourceException {
		super(session);
		this.dataSource = dataSource;

		moduleId = mapDataSourceStringToModuleId(dataSource);
	}

	@Override
	public boolean canRead(String folder, String id, String versionOrAttachmentId) throws DataSourceException {
        try {
        	final int folderId = Integer.parseInt(folder);
        	final int attachedId = Integer.parseInt(versionOrAttachmentId);
        	final int objectId = Integer.parseInt(id);

        	final AttachmentBase attachmentAccess = Attachments.getInstance();
        	return (attachmentAccess.getAttachment(session, folderId, objectId, moduleId, attachedId, session.getContext(), session.getUser(), session.getUserConfiguration()) != null);
	    } catch (final OXException e) {
	        LOG.error("DataSourceAccessCalendarTaskContact: Exception caught trying to retrieve meta data from task/calendar/contacts attachment.", e);
	        final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, false);
	        throw new DataSourceException(errorCode, e);
	    }
	}

	@Override
	public MetaData getMetaData(String folder, String id, String versionOrAttachmentId, Optional<String> authCode) throws DataSourceException {

        try {
        	final int folderId = Integer.parseInt(folder);
        	final int attachedId = Integer.parseInt(versionOrAttachmentId);
        	final int objectId = Integer.parseInt(id);

        	final AttachmentBase attachmentAccess = Attachments.getInstance();
        	final AttachmentMetadata metaData = attachmentAccess.getAttachment(session, folderId, objectId, moduleId, attachedId, session.getContext(), session.getUser(), session.getUserConfiguration());
        	if (metaData != null) {
        		final long fileSize = metaData.getFilesize();
        		final String fileName = metaData.getFilename();
        		String mimeType = metaData.getFileMIMEType();

            	// try to refine mime type using extension if we see an unknown mime type
            	final Map<String, String> docFormatInfo = DocumentFormatHelper.getFormatInfoForMimeTypeOrExtension(mimeType, FileHelper.getExtension(fileName, true));
            	if (docFormatInfo != null) {
            		mimeType = docFormatInfo.get(DocumentFormatHelper.PROP_MIME_TYPE);
            	}

            	final String hash = createHashFrom(fileSize, fileName, mimeType);
        		return new MetaData(folder, id, versionOrAttachmentId, fileSize, fileName, hash, mimeType);
        	}

        	throw new DataSourceException(ErrorCode.GENERAL_RESOURCE_NOT_FOUND_ERROR);
        } catch (final OXException e) {
            LOG.error("DataSourceAccessCalendarTaskContact: Exception caught trying to retrieve meta data from task/calendar/contacts attachment.", e);
            final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, false);
            throw new DataSourceException(errorCode, e);
        }
	}

	@Override
	public InputStream getContentStream(String folder, String id, String versionOrAttachmentId, Optional<String> authCode) throws DataSourceException {

        try {
        	final int folderId = Integer.parseInt(folder);
        	final int objectId = Integer.parseInt(id);
        	final int attachedId = Integer.parseInt(versionOrAttachmentId);
        	final AttachmentBase attachmentAccess = Attachments.getInstance();
        	final AttachmentMetadata metaData = attachmentAccess.getAttachment(session, folderId, objectId, moduleId, attachedId, session.getContext(), session.getUser(), session.getUserConfiguration());
        	if (metaData != null) {
    	        final String encryptionInfo = determineEncryptionInfo(authCode);
        		final long fileSize = metaData.getFilesize();

        		checkFilSizeAndMemoryConsumption(fileSize, encryptionInfo);

        		final InputStream inStream = attachmentAccess.getAttachedFile(session, folderId, objectId, moduleId, attachedId, session.getContext(), session.getUser(), session.getUserConfiguration());
            	final byte[] data = IOUtils.toByteArray(inStream);
            	return new ByteArrayInputStream(data);
        	}

        	throw new DataSourceException(ErrorCode.GENERAL_RESOURCE_NOT_FOUND_ERROR);
        } catch (IOException e) {
            LOG.error("DataSourceAccessCalendarTaskContact: IOException caught trying to retrieve input stream from task/calendar/contacts attachment.", e);
            throw new DataSourceException(ErrorCode.GENERAL_UNKNOWN_ERROR, e);
        } catch (final OXException e) {
            LOG.error("DataSourceAccessCalendarTaskContact: Exception caught trying to retrieve input stream from task/calendar/contacts attachment.", e);
            final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, false);
            throw new DataSourceException(errorCode, e);
        }
	}

    @Override
    public String getDataSource() {
        return dataSource;
    }

	private int mapDataSourceStringToModuleId(String source) throws DataSourceException {
		int module = -1;

		switch (source) {
		case DataSource.CALENDAR: module = Types.APPOINTMENT; break;
		case DataSource.CONTACTS: module = Types.CONTACT; break;
		case DataSource.TASKS: module = Types.TASK; break;
		default: throw new DataSourceException(ErrorCode.GENERAL_ARGUMENTS_ERROR);
		}

		return module;
	}

}
