/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.service.storage.impl;

import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;

import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageCapability;
import com.openexchange.file.storage.composition.FolderID;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.office.tools.service.config.UserConfigurationFactory;
import com.openexchange.office.tools.service.config.UserConfigurationHelper;
import com.openexchange.office.tools.service.config.UserConfigurationHelper.Mode;
import com.openexchange.office.tools.service.storage.StorageHelperService;
import com.openexchange.session.Session;

@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class StorageHelperServiceImpl implements StorageHelperService {

    // capabilities cached locally
    boolean efficientRetrieval = false;
    boolean fileVersions = false;
    boolean setCapabilitiesCalled = false;
    boolean supportsPersistenIDs = false;
    boolean ignorableVersion = false;
    boolean supportsExtendedMetaData = false;

    /**
     * Initializes a new {@link StorageHelperService}.
     * @param services
     * @param session
     * @param folderId
     */
    public StorageHelperServiceImpl(IDBasedFileAccessFactory fileFactory, UserConfigurationFactory userConfigHelper, final Session session, final String folderId) {
        final IDBasedFileAccess fileAccess = (null != session) ? fileFactory.createAccess(session) : null;
        final UserConfigurationHelper confHelper = userConfigHelper.create(session, "io.ox/office", Mode.WRITE_THROUGH);
        impl_setCapabilities(fileAccess, folderId, confHelper);
    }

    /**
     * Returns the capability for an efficient stream and meta-data retrieval.
     * @return
     */
    @Override
    public boolean supportsEfficientRetrieval() {
        return efficientRetrieval;
    }

    /**
     * Returns the capability to support file versions.
     * @return
     */
    @Override
    public boolean supportsFileVersions() {
        return fileVersions;
    }

    /**
     * Returns the capability to persistent file and folder ids.
     * @return
     */
    @Override
    public boolean supportsPersistentIDs() {
        return supportsPersistenIDs;
    }

    /**
     * Returns the capability to ignore versions, e.g. overwrite an existing
     * version.
     * @return
     */
    @Override
    public boolean supportsIgnorableVersion() {
        return ignorableVersion;
    }

    /**
     * Returns the capability to store extended meta data.
     *
     * @return <code>TRUE</code> if supported otherwise <code>FALSE</code>.
     */
    @Override
    public boolean supportsExtendedMetaData() {
        return supportsExtendedMetaData;
    }

    /**
     * Determines, using the folder id and file access, if the storage supports
     * file versions or not.
     *
     * @param fileAccess
     *  A valid file access .
     *
     * @param folderId
     *  A valid folder id to be checked for rename support or not.
     *
     * @return
     *  TRUE if the storage supports versions, otherwise not.
     */
    private boolean impl_setCapabilities(final IDBasedFileAccess fileAccess, final String folderId, final UserConfigurationHelper conf) {
        boolean result = false;

        try {
            final FolderID folderIdentifier = new FolderID(folderId);
            final String folderAccountId = folderIdentifier.getAccountId();
            final String folderService = folderIdentifier.getService();

            efficientRetrieval = fileAccess.supports(folderService, folderAccountId, FileStorageCapability.EFFICIENT_RETRIEVAL);
            fileVersions = fileAccess.supports(folderService, folderAccountId, FileStorageCapability.FILE_VERSIONS);
            supportsPersistenIDs = fileAccess.supports(folderService, folderAccountId, FileStorageCapability.PERSISTENT_IDS);
            ignorableVersion = fileAccess.supports(folderService, folderAccountId, FileStorageCapability.IGNORABLE_VERSION);
            supportsExtendedMetaData = fileAccess.supports(folderService, folderAccountId, FileStorageCapability.EXTENDED_METADATA);

            // overwrite values from configuration for QA purposes only
            final Boolean storageVersion = conf.getBoolean("module/storageFileVersions", null);
            if (null != storageVersion) {
                fileVersions = storageVersion;
            }
        } catch (NumberFormatException e) {
            // nothing to do
        } catch (OXException e) {
            // nothing to do
        }

        return result;
    }
}
