package com.openexchange.office.tools.jms;

import java.io.IOException;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.pool.PooledConnectionFactory;
import org.apache.activemq.transport.TransportListener;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.base.Throwables;
import com.openexchange.office.dcs.registry.DCSDatabase;
import com.openexchange.office.dcs.registry.DCSRegistryItem;
import com.openexchange.office.tools.common.osgi.context.OsgiBundleContextAndActivator;
import com.openexchange.office.tools.common.osgi.context.OsgiBundleContextAware;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;
import com.openexchange.office.tools.database.DatabaseException;

@Service
public class PooledConnectionFactoryCreator implements OsgiBundleContextAware, DisposableBean {

	private static final Logger log = LoggerFactory.getLogger(PooledConnectionFactoryCreator.class);
	
	@Autowired
	private JmsConfigService jmsConfigService;
	
    @Autowired
    private DCSDatabase dcsDatabase;
    
    private final ScheduledExecutorService scheduledExecService = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder("UrlInSyncWatchDogThread-%d").build());
    
    private LocalDateTime lastErrMsg = null;
	
	private PooledConnectionFactory pooledConnectionFactory = null;
	
	@Override
	public void setApplicationContext(OsgiBundleContextAndActivator bundleCtx) {
        log.debug("Creating PooledConnectionFactory for connection to JMS Server...");
        pooledConnectionFactory = new PooledConnectionFactory();
        pooledConnectionFactory.setConnectionFactory(getJmsConnectionFactory());
        bundleCtx.registerServiceToOsgi(new PooledConnectionFactoryProxy(pooledConnectionFactory), PooledConnectionFactoryProxy.class);
        bundleCtx.registerServiceToOsgi(getJmsTemplateWithoutTtl(), JmsTemplateWithoutTtl.class);
        bundleCtx.registerServiceToOsgi(getJmsTemplateWithTtl(), JmsTemplateWithTtl.class);
	}
	
    @Override
	public void destroy() throws Exception {
        scheduledExecService.shutdown();
    	//-------------------------------------------------------------------------
        log.info("... deactivate connection factory");
        if (pooledConnectionFactory != null) {
        	pooledConnectionFactory.stop();
        }
    }
	
    private JmsTemplateWithoutTtl getJmsTemplateWithoutTtl() {
    	JmsTemplateWithoutTtl res =  new JmsTemplateWithoutTtl(pooledConnectionFactory);
    	res.setTimeToLive(0);
    	res.setExplicitQosEnabled(false);
    	return res;
    }
    
    private JmsTemplateWithTtl getJmsTemplateWithTtl() {
    	JmsTemplateWithTtl res = new JmsTemplateWithTtl(pooledConnectionFactory);
    	res.setTimeToLive(60000);
    	res.setExplicitQosEnabled(true);    	
    	return res;
    }    
    
    private EnhActiveMQSSLConnectionFactory getJmsConnectionFactory() {
        final EnhActiveMQSSLConnectionFactory jmsConnectionFactory = new EnhActiveMQSSLConnectionFactory();

        String jmsUsername = jmsConfigService.getJmsUsername();
        String jmsPassword = jmsConfigService.getJmsPassword();
        if (StringUtils.isNotBlank(jmsUsername)) {
            jmsConnectionFactory.setUserName(jmsUsername);
        }
        if (StringUtils.isNotBlank(jmsPassword)) {
            jmsConnectionFactory.setPassword(jmsPassword);
        }

        final String failOverBrokerUrl = getFailoverBrokerUrl();
        jmsConnectionFactory.setBrokerURL(failOverBrokerUrl);

        if (failOverBrokerUrl.contains("ssl:")) {
            try {
                // set keystore properties
                jmsConnectionFactory.setKeyStore(jmsConfigService.getRT2SSLKeystorePath());
                jmsConnectionFactory.setKeyStorePassword(jmsConfigService.getRT2SSLKeystorePassword());

                // set truststore properties
                jmsConnectionFactory.setTrustStore(jmsConfigService.getRT2SSLTrustStorePath());
                jmsConnectionFactory.setTrustStorePassword(jmsConfigService.getRT2SSLTrustStorePassword());
            } catch (Exception e) {
                log.error(Throwables.getRootCause(e).getMessage());
            }
        }

        jmsConnectionFactory.setTransportListener(new TransportListener() {

            private String lastErrorMsg = null;

            @Override
            public void transportResumed() {
                log.info("Resumed transport connection to dcs with broker url {}", jmsConnectionFactory.getBrokerURL());
            }

            @Override
            public void transportInterupted() {
            }

            @Override
            public void onException(IOException ex) {
                if ((lastErrorMsg == null) || !lastErrorMsg.equals(ex.getMessage())) {
                    log.info("Caught IOException: {}", ex);
                    lastErrorMsg = ex.getMessage();
                }
                jmsConnectionFactory.setBrokerURL(getFailoverBrokerUrl());
            }

            @Override
            public void onCommand(Object cmd) {
                log.debug("onCommand: {}", cmd);
            }
        });
        scheduledExecService.scheduleWithFixedDelay(new UrlInSyncWatchDogThread(jmsConnectionFactory), 1, 1, TimeUnit.MINUTES);
        return jmsConnectionFactory;
    }
	
    private String getFailoverBrokerUrl() {
        List<DCSRegistryItem> dcsRegistryItems = new ArrayList<>();
        try {
             dcsRegistryItems = dcsDatabase.getRegisteredDCSItems();
        } catch (DatabaseException ex) {
            if (lastErrMsg == null) {
                lastErrMsg = LocalDateTime.now();
                log.error("DatabaseException caught - cannot provide broker url, please check configuration & setup! DatabaseException: " + ex.getMessage());
            } else {
                if (ChronoUnit.MINUTES.between(LocalDateTime.now(), lastErrMsg) > 1) {
                    log.error("DatabaseException caught - cannot provide broker url, please check configuration & setup!");
                    lastErrMsg = null;
                }
            }
        }

        final StringBuilder res = new StringBuilder(256).append("failover:(");

        for (int i=0;i<dcsRegistryItems.size();++i) {
            final DCSRegistryItem item = dcsRegistryItems.get(i);
            final boolean useSSL = item.isUseSSL();

            res.append(useSSL ? "ssl" : "tcp").append("://").
                append(item.getInterface()).
                append(":").
                append(item.getJMSPort());

            if (useSSL) {
                res.append("?verifyHostName=" + (jmsConfigService.isRT2SSLVerifyHostname() ? "true" : "false"));
            }

            if ((i + 1) < dcsRegistryItems.size()) {
                res.append(",");
            }
        }
        res.append(")?maxReconnectAttempts=3");
        ActiveMQConnectionFactory amqFactory = (ActiveMQConnectionFactory)(pooledConnectionFactory.getConnectionFactory());
        if ((amqFactory == null) || !amqFactory.getBrokerURL().equals(res.toString())) {
            log.info("Broker Url for the connection to DCS is: {}", res);
        } else {
            log.debug("Broker Url for the connection to DCS is: {}", res);
        }
        return res.toString();
    }    
    
    private class UrlInSyncWatchDogThread implements Runnable {

        private ActiveMQConnectionFactory activeMqConnectionFactory;

        public UrlInSyncWatchDogThread(ActiveMQConnectionFactory activeMqConnectionFactory) {
            this.activeMqConnectionFactory = activeMqConnectionFactory;
        }

        @Override
        public void run() {
            String currentUrl = activeMqConnectionFactory.getBrokerURL();
            String dbBasedUrl = getFailoverBrokerUrl();
            if (!currentUrl.equals(dbBasedUrl)) {
                activeMqConnectionFactory.setBrokerURL(dbBasedUrl);
            }
        }
    }    
}
