package com.openexchange.office.tools.common.threading;

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.Locale;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.lang3.Validate;

import com.openexchange.logging.MDCEnabledThreadGroup;

public class ThreadFactoryBuilder {
	
	  public static final MDCEnabledThreadGroup MDC_ENABLED_THREAD_GROUP = new MDCEnabledThreadGroup();
	
	  private String nameFormat = null;
	  private Boolean daemon = null;
	  private UncaughtExceptionHandler uncaughtExceptionHandler = null;
	  private ThreadGroup threadGroup = null;

	  public ThreadFactoryBuilder() {}

	  public ThreadFactoryBuilder(String nameFormat) {
		  format(nameFormat, 0); // fail fast if the format is bad or null
		  this.nameFormat = nameFormat;
		  this.threadGroup = MDC_ENABLED_THREAD_GROUP;		    
	  }
	  
	  public ThreadFactoryBuilder setNameFormat(String nameFormat) {
	    format(nameFormat, 0); // fail fast if the format is bad or null
	    this.nameFormat = nameFormat;
	    return this;
	  }

	  public ThreadFactoryBuilder setDaemon(boolean daemon) {
	    this.daemon = daemon;
	    return this;
	  }

	  public ThreadFactoryBuilder setThreadGroup(ThreadGroup threadGroup) {
		  this.threadGroup = Validate.notNull(threadGroup);
		  return this;
	  }	  
	  
	  public ThreadFactoryBuilder setUncaughtExceptionHandler(UncaughtExceptionHandler uncaughtExceptionHandler) {
	    this.uncaughtExceptionHandler = Validate.notNull(uncaughtExceptionHandler);
	    return this;
	  }

	  public ThreadFactory build() {
	    return doBuild(this);
	  }

	  // Split out so that the anonymous ThreadFactory can't contain a reference back to the builder.
	  // At least, I assume that's why. TODO(cpovirk): Check, and maybe add a test for this.
	  private static ThreadFactory doBuild(ThreadFactoryBuilder builder) {
	    final String nameFormat = builder.nameFormat;
	    final Boolean daemon = builder.daemon;
	    final UncaughtExceptionHandler uncaughtExceptionHandler = builder.uncaughtExceptionHandler;
	    final AtomicLong count = (nameFormat != null) ? new AtomicLong(0) : null;
	    final ThreadGroup threadGroup = builder.threadGroup;
	    return new ThreadFactory() {
	      @Override
	      public Thread newThread(Runnable runnable) { 
	        Thread thread;
	        if (threadGroup == null) {
	        	thread = new Thread(runnable);
	        } else {
	        	thread = new Thread(threadGroup, runnable);
	        }
	        if (nameFormat != null) {
	          thread.setName(format(nameFormat, count.getAndIncrement()));
	        }
	        if (daemon != null) {
	          thread.setDaemon(daemon);
	        }
	        if (uncaughtExceptionHandler != null) {
	          thread.setUncaughtExceptionHandler(uncaughtExceptionHandler);
	        }
	        return thread;
	      }
	    };
	  }

	  private static String format(String format, Object... args) {
	    return String.format(Locale.ROOT, format, args);
	  }
}
