/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.rest;

import javax.servlet.http.HttpServletRequest;

import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestDataTools;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.office.rt2.core.proxy.RT2DocProxy;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyRegistry;
import com.openexchange.office.rt2.core.ws.RT2ChannelId;
import com.openexchange.office.rt2.core.ws.RT2EnhDefaultWebSocket;
import com.openexchange.office.rt2.core.ws.RT2WSApp;
import com.openexchange.office.rt2.core.ws.RT2WebSocketListener;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageFactory;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.error.HttpStatusCode;
import com.openexchange.office.tools.service.session.SessionService;
import com.openexchange.tools.session.ServerSession;

@RestController
public class EmergencyLeaveAction implements AJAXActionService {

    // ---------------------------------------------------------------
    private static final Logger log = LoggerFactory.getLogger(EmergencyLeaveAction.class);    
    
    // ---------------------------------------------------------------
    private static final String[] aMandatoryParams = { RESTParameters.PARAMETER_CLIENTUID,
                                                       RESTParameters.PARAMETER_DOCUID,
    		                                           RESTParameters.PARAMETER_CHANNELUID };
    
    public static final String WS_URL_ROOT = "/rt2/v1/default/";
    
    @Autowired
    private SessionService sessionService;

    @Autowired
    private RT2WSApp rt2WSApp;
    
    @Autowired
    private RT2WebSocketListener rt2WebSocketListener;
    
    @Autowired
    private RT2DocProxyRegistry rt2DocProxyRegistry;
    
    // ---------------------------------------------------------------
	@Override
	public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {

		if (!ParamValidator.areAllParamsNonEmpty(requestData, aMandatoryParams))
            return ParamValidator.getResultFor(HttpStatusCode.BAD_REQUEST.getStatusCode());

        final String channelUID = requestData.getParameter(RESTParameters.PARAMETER_CHANNELUID);
        
        final String channelUrl = WS_URL_ROOT + channelUID;
        final String clientUID = requestData.getParameter(RESTParameters.PARAMETER_CLIENTUID);
        final String docUID = requestData.getParameter(RESTParameters.PARAMETER_DOCUID);
        final HttpServletRequest servletRequest = requestData.optHttpServletRequest();

        JSONObject leaveData = (JSONObject) requestData.getData();
        if (null == leaveData && (servletRequest != null)) {
        	// try to extract body data using a different approach (e.g.
        	// support POST with preferStream = false)
            requestData.setUploadStreamProvider(null);
            AJAXRequestDataTools.loadRequestBody(requestData);
            leaveData = (JSONObject) requestData.getData();
        }

    	try {
    		if (leaveData == null) {
    			leaveData = new JSONObject();
    		}

            log.debug("RT2: emergencyLeave request received for docUID={}, clientUID={}", docUID, clientUID);

            final RT2EnhDefaultWebSocket clientConnection = rt2WSApp.getWebSocketWithId(new RT2ChannelId(channelUrl));
            if (null != clientConnection) {
            	rt2WebSocketListener.emergencyLeave(clientConnection, docUID, clientUID, session.getSessionID(), leaveData);
            } else {
            	sendEmergencyLeaveViaDocProxy(session, new RT2CliendUidType(clientUID), new RT2DocUidType(docUID), leaveData);
            }
    	} catch (final JSONException e) {
    		log.info("RT2: JSONException caught trying to decode leave data from emergency request", e);
    		return new AJAXRequestResult(ErrorCode.GENERAL_ARGUMENTS_ERROR.getAsJSONResultObject());
    	} catch (final DocProxyNotFoundException e) {
    		log.info("RT2: DocProxyNotFoundException caught trying trying to send emergency request", e);
    		return new AJAXRequestResult(ErrorCode.GENERAL_CLIENT_UID_UNKNOWN_ERROR.getAsJSONResultObject());
    	} catch (final ServiceNotAvailableException e) {
    		log.error("RT2: ServiceNotAvailableException caught trying trying to send emergency request", e);
    		return new AJAXRequestResult(ErrorCode.GENERAL_MISSING_SERVICE_ERROR.getAsJSONResultObject());
    	} catch (final Exception e) {
    		log.warn("RT2: Exception caught trying to send emergency request", e);
    		return new AJAXRequestResult(ErrorCode.GENERAL_UNKNOWN_ERROR.getAsJSONResultObject());
    	}

        return ParamValidator.getResultFor(HttpStatusCode.OK.getStatusCode());
	}

    // ---------------------------------------------------------------
	private void sendEmergencyLeaveViaDocProxy(final ServerSession session, RT2CliendUidType clientUID, RT2DocUidType docUID, final JSONObject leaveData) throws Exception {
    	final String docProxyID = RT2DocProxy.calcID(clientUID, docUID);
    	final RT2DocProxy docProxy = rt2DocProxyRegistry.getDocProxy(docProxyID);
    	if (docProxy == null) {
    		throw new DocProxyNotFoundException("RT2DocProxy " + docProxyID + " not found in registry - must drop emergency leave message");
    	}

  		final RT2Message aEmergencyLeaveRequest = RT2MessageFactory.newMessage(RT2MessageType.REQUEST_EMERGENCY_LEAVE, clientUID, docUID);
		RT2MessageGetSet.setSessionID(aEmergencyLeaveRequest, sessionService.getSessionId(session));
		aEmergencyLeaveRequest.setBody(leaveData);
	
    	docProxy.addMsgFromWebSocket(aEmergencyLeaveRequest);
        docProxy.leave(aEmergencyLeaveRequest);
	}
}
