/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.protocol;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class RT2MessageFormatter {

	public static final String MSG_TYPE = "msg_type";

    public static final List<String> DEFAULT_LOG_FORMAT;
    static {
        final List<String> tmp = new ArrayList<>();
        tmp.add(RT2Protocol.HEADER_MSG_ID);
        tmp.add(RT2Protocol.HEADER_CLIENT_UID);
        tmp.add(RT2Protocol.HEADER_DOC_UID);
        tmp.add(MSG_TYPE);
        tmp.add(RT2Protocol.HEADER_SEQ_NR);

        DEFAULT_LOG_FORMAT = Collections.unmodifiableList(tmp);
    }


	private RT2MessageFormatter() {
		// nothing to do
	}

	public static String formatForDefaultOutput(RT2Message msg) {
		final StringBuilder tmp = new StringBuilder("RT2Message: ");
		DEFAULT_LOG_FORMAT.stream().forEach(p -> {
			tmp.append(p);
			tmp.append(": ");
			tmp.append(getMessageValueFromName(msg, p));
			tmp.append(", ");
		});
		return tmp.toString();
	}

	public static String formatForDefaultOutput(Collection<RT2Message> msgs) {
		final StringBuilder tmp = new StringBuilder("List of RT2Messages: ");
		msgs.stream().forEach(m -> {
			tmp.append(formatForDefaultOutput(m));
		});
		return tmp.toString();
	}

	public static String formatForOutput(RT2Message msg, String... propsToOutput) {
		StringBuilder tmp = new StringBuilder("RT2Message: ");
		for (String p : propsToOutput)
			appendMessageValue(tmp, msg, p);
		return tmp.toString();
	}

	public static String formatForOutput(Collection<RT2Message> msgs, String... propsToOutput) {
		final StringBuilder tmp = new StringBuilder("List of RT2Messages: ");
		msgs.stream().forEach(m -> {
			tmp.append(formatForOutput(m, propsToOutput));
		});
		return tmp.toString();
	}

	public static String formatForOutput(Collection<RT2Message> msgs, int maxCount, String... propsToOutput) {
		final StringBuilder tmp = new StringBuilder("List of non-ACKed RT2Messages: ");
		msgs.stream().limit(Math.min(msgs.size(), maxCount))	
		             .forEach(m -> { 
		                  tmp.append(formatForOutput(m, propsToOutput));
		              });
		return tmp.toString();
	}

	private static void appendMessageValue(StringBuilder b, RT2Message msg, String propName) {
		b.append(propName);
		b.append(": ");
		b.append(getMessageValueFromName(msg, propName));
		b.append(", ");
	}

	private static String getMessageValueFromName(RT2Message msg, String propName) {
		if (MSG_TYPE.equalsIgnoreCase(propName))
			return msg.getType().toString();

		return toString(msg.getHeaderUnchecked(propName));
	}

	private static String toString(Object o) {
		return (o == null) ? "null" : o.toString();
	}
}
