/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.hazelcast;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.osgi.framework.BundleException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.hazelcast.core.IMap;
import com.openexchange.exception.OXException;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeHealthPredicate;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeHealthState;
import com.openexchange.office.rt2.hazelcast.serialization.PortableNodeHealthStatePredicate;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.osgi.ExceptionUtils;

@Service
@RegisteredService(registeredClass=RT2NodeHealthMap.class)
public class HzNodeHealthMap extends RT2HazelcastService implements RT2NodeHealthMap
{
	private static final Logger log = LoggerFactory.getLogger(HzNodeHealthMap.class);

    @Override
	public void afterPropertiesSet() throws Exception {
    	mapIdentifier = HzHelper.discoverMapName(hzInstance.getConfig(), "rt2NodeHealthMap-");
        if(StringUtils.isEmpty(mapIdentifier)) {
            final String msg = "Distributed rt2 node health map couldn't be found in hazelcast configuration";
            throw new IllegalStateException(msg, new BundleException(msg, BundleException.ACTIVATOR_ERROR));
        }
        log.info("Registered rt2NodeHealthMap to Hazelcast");
	}


    //-------------------------------------------------------------------------
	@Override
	public RT2NodeHealthState get(String sNodeUID) throws OXException {
		RT2NodeHealthState aNodeHealthState = null;

        IMap<String, PortableNodeHealthState> allNodeHealthStates = getHealthStateMapping();
        PortableNodeHealthState aPortableNodeHealthState = allNodeHealthStates.get(sNodeUID);
        if (aPortableNodeHealthState != null) {
        	aNodeHealthState = PortableNodeHealthState.createFrom(aPortableNodeHealthState);
        }

        return aNodeHealthState;
	}

    //-------------------------------------------------------------------------
	@Override
	public Map<String, RT2NodeHealthState> get(Collection<String> nodeUIDs) throws OXException {
        final Map<String, RT2NodeHealthState> aFoundNodesMap = new HashMap<>();
        final Set<String> aNodeUIDSet = new HashSet<>();

        for (final String sNodeUID : nodeUIDs) {
        	aNodeUIDSet.add(sNodeUID);
        }

        if (!aNodeUIDSet.isEmpty()) {
            IMap<String, PortableNodeHealthState> allResources = getHealthStateMapping();
            Map<String, PortableNodeHealthState> aMatchNodeHealthStates = allResources.getAll(aNodeUIDSet);
            if (aMatchNodeHealthStates != null) {
                for (Entry<String, PortableNodeHealthState> entry : aMatchNodeHealthStates.entrySet()) {
                    final String sNodeUID = entry.getKey();
                    final PortableNodeHealthState aFoundNodeHealthState = entry.getValue();
                    aFoundNodesMap.put(sNodeUID, PortableNodeHealthState.createFrom(aFoundNodeHealthState));
                }
            }
        }

        return aFoundNodesMap;
	}

    //-------------------------------------------------------------------------
	@Override
	public RT2NodeHealthState set(String sNodeUID, RT2NodeHealthState state) throws OXException {
        final PortableNodeHealthState aCurrPortableNodeHealthState = new PortableNodeHealthState(state);

        PortableNodeHealthState aPrevPortableNodeHealthState = null;
        RT2NodeHealthState      aPrevNodeHealthStates        = null;

        try {
            final IMap<String, PortableNodeHealthState> allNodeHealthStates = getHealthStateMapping();
           	aPrevPortableNodeHealthState = allNodeHealthStates.put(sNodeUID, aCurrPortableNodeHealthState);
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }

        if (null != aPrevPortableNodeHealthState) {
        	aPrevNodeHealthStates = PortableNodeHealthState.createFrom(aPrevPortableNodeHealthState);
        }

        return aPrevNodeHealthStates;
	}

    //-------------------------------------------------------------------------
	@Override
	public void setIfAbsent(String sNodeUID, RT2NodeHealthState state) throws OXException {
        final PortableNodeHealthState aCurrPortableNodeHealthState = new PortableNodeHealthState(state);
        try {
            final IMap<String, PortableNodeHealthState> allNodeHealthStates = getHealthStateMapping();
            allNodeHealthStates.putIfAbsent(sNodeUID, aCurrPortableNodeHealthState);
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }
	}

    //-------------------------------------------------------------------------
	@Override
	public RT2NodeHealthState remove(String sNodeUID) throws OXException {
		PortableNodeHealthState aPrevPortableNodeHealthState = null;
		RT2NodeHealthState      aPrevNodeHealthState         = null;

        try {
            final IMap<String, PortableNodeHealthState> allDocStates = getHealthStateMapping();
            aPrevPortableNodeHealthState = allDocStates.remove(sNodeUID);
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }

        if (null != aPrevPortableNodeHealthState) {
        	aPrevNodeHealthState = PortableNodeHealthState.createFrom(aPrevPortableNodeHealthState);
        }

        return aPrevNodeHealthState;
	}

    //-------------------------------------------------------------------------
	@Override
	public Map<String, RT2NodeHealthState> remove(Collection<String> aNodeUIDs) throws OXException {
        final Map<String, RT2NodeHealthState> aRemovedMap = new HashMap<>();
        final Set<String>                     aNodeUIDSet = new HashSet<>();

        for (final String sNodeUID : aNodeUIDs) {
        	aNodeUIDSet.add(sNodeUID);
        }

        if (!aNodeUIDSet.isEmpty()) {
            IMap<String, PortableNodeHealthState> aAllNodeHealthStates           = getHealthStateMapping();
            Map<String, PortableNodeHealthState>  aMatchPortableNodeHealthStates = aAllNodeHealthStates.getAll(aNodeUIDSet);

            if (aMatchPortableNodeHealthStates != null) {
                for (Entry<String, PortableNodeHealthState> entry : aMatchPortableNodeHealthStates.entrySet()) {
                    final String sFoundUID = entry.getKey();
                    final PortableNodeHealthState aFoundNodeHealthState = entry.getValue();
                    aRemovedMap.put(sFoundUID, PortableNodeHealthState.createFrom(aFoundNodeHealthState));

                    aAllNodeHealthStates.remove(sFoundUID);
                }
            }
        }

        return aRemovedMap;
	}

    //-------------------------------------------------------------------------
    /**
     * Find all member nodes in this directory that are clean-up by a given member node.
     *
     * @param member The cluster member
     * @return all Resources in this directory that are located on the given member node.
     * @throws OXException
     */
    @Override
    public Set<RT2NodeHealthState> getCleanupNodesOfMember(final String sNodeUUID)throws OXException {
        final IMap<String, PortableNodeHealthState>       allResources  = getHealthStateMapping();
        final PortableNodeHealthPredicate                 nodePredicate = new PortableNodeHealthPredicate(sNodeUUID);
        final Set<Entry<String, PortableNodeHealthState>> matching      = allResources.entrySet(nodePredicate);

        final Set<RT2NodeHealthState> foundIds = new HashSet<>();
        final Iterator<Entry<String, PortableNodeHealthState>> iterator = matching.iterator();
        while(iterator.hasNext()) {
            try {
                final Entry<String, PortableNodeHealthState> next = iterator.next();
                foundIds.add(PortableNodeHealthState.createFrom(next.getValue()));
            } catch (Exception e) {
            	log.error("Couldn't add node to cleanup that was found for member " + sNodeUUID, e);
            }
        }

        return foundIds;
    }

    //-------------------------------------------------------------------------
    @Override
    public Set<RT2NodeHealthState> getMembersOfState(final String sState) {
        final IMap<String, PortableNodeHealthState> allResources   = getHealthStateMapping();
        final PortableNodeHealthStatePredicate statePredicate      = new PortableNodeHealthStatePredicate(sState);
        final Set<Entry<String, PortableNodeHealthState>> matching = allResources.entrySet(statePredicate);

        final Set<RT2NodeHealthState> foundIds = new HashSet<>();
        final Iterator<Entry<String, PortableNodeHealthState>> iterator = matching.iterator();
        while(iterator.hasNext()) {
            try {
                final Entry<String, PortableNodeHealthState> next = iterator.next();
                foundIds.add(PortableNodeHealthState.createFrom(next.getValue()));
            } catch (Exception e) {
            	log.error("Couldn't add member that was found for state " + sState, e);
            }
        }

        return foundIds;
    }

    //-------------------------------------------------------------------------
    @Override
    public Set<RT2NodeHealthState> getAllMembers() {
        final IMap<String, PortableNodeHealthState> allResources = getHealthStateMapping();

        final Set<RT2NodeHealthState> result = new HashSet<>();
        allResources.forEach((s, p) -> {
            result.add(PortableNodeHealthState.createFrom(p));
        });

        return result;
    }

    //-------------------------------------------------------------------------
    @Override
    public String getUniqueMapName() {
        return mapIdentifier;
	}

    //-------------------------------------------------------------------------
    public IMap<String, PortableNodeHealthState> getHealthStateMapping() {
        return hzInstance.getMap(mapIdentifier);
    }
}
