/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.sequence;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageGetSet;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.tools.common.TimeStampUtils;

public class ClientState {
	private static final Logger log = LoggerFactory.getLogger(ClientState.class);

	private static class BackupMessageData {
		private RT2Message backupMessage;
		private long timeStamp;

		BackupMessageData(RT2Message msg) {
			backupMessage = msg;
			timeStamp = System.currentTimeMillis();
		}

		private static int oldest(BackupMessageData b1, BackupMessageData b2) {
			return (b1.timeStamp < b2.timeStamp) ?  -1 : ((b1.timeStamp == b2.timeStamp) ? 0 : 1);
		}
	}

	private RT2CliendUidType clientUID;
	private AtomicBoolean online = new AtomicBoolean(true);
	private Map<Integer, BackupMessageData> msgBackup = Collections.synchronizedMap(new HashMap<>());
	private Set<Integer> ackSet = Collections.synchronizedSet(new HashSet<>());
	private AtomicLong lastMsgReceived = new AtomicLong(0);
	private AtomicLong lastAckMsgReceived = new AtomicLong(0);
	private AtomicLong onOfflineTimeStamp = new AtomicLong(0);
	
	//-------------------------------------------------------------------------
	public ClientState(final RT2CliendUidType clientUID, boolean online) {
		this.clientUID = clientUID;
		this.online.set(online);
		this.lastMsgReceived.set(TimeStampUtils.NO_TIMESTAMP);
		this.onOfflineTimeStamp.set(System.currentTimeMillis());
	}

	//-------------------------------------------------------------------------
	public boolean isOnline() {
		return online.get();
	}

	//-------------------------------------------------------------------------
	public void setOnlineState(boolean bNewState) {
		if (online.compareAndSet(!bNewState, bNewState)) {
			onOfflineTimeStamp.set(System.currentTimeMillis());
		}
	}

	//-------------------------------------------------------------------------
	public void addReceivedSeqNrCol(Collection<Integer> seqNrCol) {
		ackSet.addAll(seqNrCol);
	}

	//-------------------------------------------------------------------------	
	public Set<RT2Message> getBackupMsgs() {
		synchronized(msgBackup) {
			return msgBackup.values().stream()
	                 .map(b -> b.backupMessage)
	                 .collect(Collectors.toSet());
		}
	}

	//-------------------------------------------------------------------------
	public int getMsgBackupSize() {
		return msgBackup.size();
	}
	
	//-------------------------------------------------------------------------
	public Set<Integer> getAckSet() {
		synchronized (ackSet) {
			final Set<Integer> aNextAckSet = ackSet;
			ackSet = Collections.synchronizedSet(new HashSet<>());
			return aNextAckSet;
		}
	}

	//-------------------------------------------------------------------------
	public long onlineSince() {
		return online.get() ? onOfflineTimeStamp.get() : TimeStampUtils.NO_TIMESTAMP;
	}

	//-------------------------------------------------------------------------
	public long offlineSince() {
		return (!online.get()) ? onOfflineTimeStamp.get() : TimeStampUtils.NO_TIMESTAMP;
	}

	//-------------------------------------------------------------------------
	public long getLastMsgReceived() {
		return lastMsgReceived.get();
	}
	
	//-------------------------------------------------------------------------
	public void updateLastMsgReceived() {
		this.lastMsgReceived.set(System.currentTimeMillis());
	}

	//-------------------------------------------------------------------------
	public long getLastAckMsgReceived() {
		return lastAckMsgReceived.get();
	}

	//-------------------------------------------------------------------------
	public void updateLastAckMsgReceived() {
		lastAckMsgReceived.set(System.currentTimeMillis());
	}

	//-------------------------------------------------------------------------
	public long getOldestBackupMessageTimeStamp() {
		synchronized (msgBackup) {
			final Optional<BackupMessageData> oldestBackupMsg = 
					msgBackup.values().stream().min(BackupMessageData::oldest);
			return oldestBackupMsg.isPresent() ? oldestBackupMsg.get().timeStamp : TimeStampUtils.NO_TIMESTAMP;
		}
	}

	//-------------------------------------------------------------------------
	public void backupMessage(final RT2Message rMsg) {
	    final Integer nSeqNr = RT2MessageGetSet.getSeqNumber(rMsg);
	    
		final BackupMessageData rOld = msgBackup.put(nSeqNr, new BackupMessageData(rMsg));

		if (rOld != null)
			log.error("RT2: Message with seq-nr " + nSeqNr + " already stored!");
	}

	//-------------------------------------------------------------------------
	public void removeStoredMessages(List<Object> ackList) {
		synchronized (msgBackup) {
			msgBackup.keySet().removeAll(ackList);
		}
	}

	//-------------------------------------------------------------------------
	public void removeStoredMessages(int start, int end) {
		synchronized (msgBackup) {
			msgBackup.keySet().removeIf(k -> (k >= start && k <= end));
		}
	}

	//-------------------------------------------------------------------------
	public List<RT2Message> getRequestedMessages(List<Object> aNackList) {
		List<RT2Message> lResult = new ArrayList<>();
		for (Object i : aNackList) {
			final BackupMessageData msgData = msgBackup.get(i);
			if (msgData != null)
				lResult.add(msgData.backupMessage);
			else {
				lResult.clear();
				log.error("RT2: Couldn't find requested message with seq-nr {} for client UID {}. Existing entries: {}", i, clientUID, msgBackup.keySet());
				break;
			}
		}

		return lResult;
	}
}
