package com.openexchange.office.rt2.core.sequence;

import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.RT2MessageLoggerService;
import com.openexchange.office.rt2.core.RT2MessageSender;
import com.openexchange.office.rt2.core.cache.RT2HazelcastHelperService;
import com.openexchange.office.rt2.core.doc.ClientEventData;
import com.openexchange.office.rt2.core.doc.IClientEventListener;
import com.openexchange.office.rt2.core.doc.IDocProcessorEventListener;
import com.openexchange.office.rt2.core.exception.RT2InvalidClientUIDException;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.annotation.Async;

@Service
@Async(initialDelay=ClientLeftRemovedService.FREQ_CLEANUP_CLIENTS_LEFT, period=ClientLeftRemovedService.FREQ_CLEANUP_CLIENTS_LEFT, timeUnit=TimeUnit.MILLISECONDS)
public class ClientLeftRemovedService implements Runnable, IDocProcessorEventListener, IClientEventListener {
	
	private static final Logger log = LoggerFactory.getLogger(ClientLeftRemovedService.class);
	
	//-------------------------------------------------------------------------	
	public static final long TIME_CLEANUP_CLIENTS_LEFT = 300000;
	
	public static final long FREQ_CLEANUP_CLIENTS_LEFT = 60000;

	//--------------------------Services--------------------------------------	
	@Autowired
	private ClientSequenceQueueDisposerService clientSequenceQueueDisposerService;
	
	@Autowired
	private RT2MessageSender messageSender;
	
	@Autowired
	private RT2MessageLoggerService messageLoggerService;
	
	@Autowired
	private RT2HazelcastHelperService rt2HazelcastHelperService;
	
	//-------------------------------------------------------------------------
	private final ConcurrentHashMap<RT2DocUidType, Map<RT2CliendUidType, Long>> clientsLeft = new ConcurrentHashMap<>();

	//-------------------------------------------------------------------------	
	public void addClientLeft(RT2DocUidType docUid, RT2CliendUidType clientUid) {
		getClientMapOfDoc(docUid).put(clientUid, System.currentTimeMillis());
	}
	
	//-------------------------------------------------------------------------
	public long getClientLeft(RT2DocUidType docUid, RT2CliendUidType clientUid) {
		Map<RT2CliendUidType, Long> clientMap = getClientMapOfDoc(docUid);
		if (clientMap.containsKey(clientUid)) {
				return clientMap.get(clientUid);
		}
		return -1l;
	}
	
	//-------------------------------------------------------------------------	
    public boolean checkStateIfClientUnknown(final RT2DocUidType docUid, final RT2CliendUidType sClientUID, final RT2Message msgFromUnknownClient) throws RT2InvalidClientUIDException {
    	final RT2MessageType msgType = msgFromUnknownClient.getType();
    	
		switch (msgType) {
			case ACK_SIMPLE:
			case NACK_SIMPLE: {
				// We ignore low-level msgs if the client uid is not known - this
				// could be a race of msgs where a client is about to leave and we
				// already received the last msg, but the leave-process is still in
				// progress - therefore the RT2DocProxy can send msgs to us!
				return true;
			}
			case REQUEST_CLOSE_DOC:
			case REQUEST_LEAVE: {
				long timeLeft = getClientLeft(docUid, sClientUID);
				if (timeLeft >= 0) {
					// We handle messages from clients that try to close/leave although they
					// already left the document instance more gracefully. This can happen if
					// we close/leave a client due to previous errors which the client hasn't
					// received yet.
					final Date date = new Date(timeLeft);
					log.debug("RT2: received message {} with client uid {} that left already at {} - just ignore message", msgFromUnknownClient, sClientUID, SimpleDateFormat.getDateInstance().format(date));
					return true;
				}
				break;
			}
			default: { /* nothing to do - throws exception */ } break;
			}

		log.warn("RT2: received message {} with unknown client uid {}", msgFromUnknownClient, sClientUID);
		throw new RT2InvalidClientUIDException(messageLoggerService.getMsgsAsList(docUid));
    }
		
	//-------------------------------------------------------------------------	
	@Override
	public void run() {
		checkIfClientHasToBeRemoved();
		checkForGapsInMessages();
	}

	private void checkForGapsInMessages() {
		clientsLeft.forEach((docUid, m) -> {
			for (RT2CliendUidType clientUid : clientSequenceQueueDisposerService.determineReceivers(docUid, null)) {
				try {					
					ClientSequenceQueue clientSequenceQueue = clientSequenceQueueDisposerService.getClientSequenceQueueForClient(docUid, clientUid, false);
					if (clientSequenceQueue != null) {
						if (clientSequenceQueue.hasQueuedInMessages()) {
							RT2Message nackMsg = clientSequenceQueueDisposerService.checkForAndRequestMissingMessagesViaNack(docUid, clientUid);
							if (nackMsg != null) {
								messageSender.sendMessageTo(clientUid, nackMsg);
							}
						}
					}
				} catch (Exception ex) {
					log.warn(ex.getMessage());
				}
			}			
		});
	}
	
	private void checkIfClientHasToBeRemoved() {
		try {
	        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "ClientLeftRemoveThread");
	        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, rt2HazelcastHelperService.getHazelcastLocalNodeUuid());        												
	        
			final long now = System.currentTimeMillis();

			clientsLeft.values().forEach(c -> {
				Set<RT2CliendUidType> clientsToCleanup = c.keySet()
						.stream()
						.filter(k -> { return needToCleanupClientUID(c.get(k), now); })
						.collect(Collectors.toSet());
				clientsToCleanup.forEach(k -> { c.remove(k); });
				if ((clientsToCleanup != null) && (!clientsToCleanup.isEmpty())) {
					log.debug("RT2: removing {} left client uids after timeout", clientsToCleanup.size());
				}				
			});
		} catch (Exception e) {
            log.warn("RT2: Exception caught while trying to lazy remove stored left client uids", e);
		}
	}

	//-------------------------------------------------------------------------
	private static boolean needToCleanupClientUID(final Long time, long now) {
		return (time != null) ? ((now - time) >= TIME_CLEANUP_CLIENTS_LEFT) : false;
	}
	
	@Override
	public void docCreated(RT2DocUidType docUid) {
		if (clientsLeft.put(docUid, Collections.synchronizedMap(new HashMap<>())) != null) {
			log.warn("New doc added, but old doc with id {} found! Overwrite it!", docUid);
		}
	}

	@Override
	public void docDisposed(RT2DocUidType docUid) {
		clientsLeft.remove(docUid);
	}

	@Override
	public void clientAdded(ClientEventData clientEventData) {
	}

	@Override
	public void clientRemoved(ClientEventData clientEventData) {
		if (clientsLeft.contains(clientEventData.getDocUid())) {
			clientsLeft.get(clientEventData.getDocUid()).remove(clientEventData.getClientUid());
		}
	}
	
	private Map<RT2CliendUidType, Long> getClientMapOfDoc(RT2DocUidType docUid) {
		Map<RT2CliendUidType, Long> newMap = Collections.synchronizedMap(new HashMap<>());
		Map<RT2CliendUidType, Long> oldMap = clientsLeft.putIfAbsent(docUid, newMap);
		if (oldMap != null) {
			newMap = oldMap;
		}
		return newMap;
	}
}
