package com.openexchange.office.rt2.core.proxy;

import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.hazelcast.core.HazelcastInstance;
import com.openexchange.office.rt2.core.cache.RT2DocInfo;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.tools.annotation.RegisteredService;

@Service
@RegisteredService
public class RT2DocInfoRegistry {

	private static final Logger log = LoggerFactory.getLogger(RT2DocInfoRegistry.class);

    //-------------------------------------------------------------------------
	@Autowired
    private HazelcastInstance hzInstance;    
	
    //-------------------------------------------------------------------------
    private final ConcurrentHashMap< RT2DocUidType, RT2DocInfo > docInfoRegistry = new ConcurrentHashMap<>();

    public void clear() {
        docInfoRegistry.clear ();
    }

	public RT2DocInfo getDocInfoWithoutCreate(final RT2DocUidType sDocUID) {
		RT2DocInfo aInfo = docInfoRegistry.get(sDocUID);
		return aInfo;
	}
    
    
    //-------------------------------------------------------------------------
	/** @return an object knowing all data about a document (does not matter if it's open or not).
	 *
	 *	@param	docUID [IN]
	 *			the unique ID for the document where those data object is requested for.
	 */
	public RT2DocInfo getDocInfo (final RT2DocUidType docUID) {
		
		RT2DocInfo aInfo = docInfoRegistry.putIfAbsent(docUID, new RT2DocInfo (hzInstance, docUID.getValue()));
		if (aInfo != null) {
			return aInfo;
		}
		return docInfoRegistry.get(docUID);
	}

    //-------------------------------------------------------------------------
	/** @return an object knowing all data about a document (does not matter if it's open or not).
	 *          The object is NOT added to the internal map if the doc-uid is unknown. The object
	 *          is for information purposes only.
	 *
	 *	@param	docUID [IN]
	 *			the unique ID for the document where those data object is requested for.
	 */
	public RT2DocInfo peekDocInfo (final RT2DocUidType docUID) {
        RT2DocInfo info = docInfoRegistry.get(docUID);

        if (info == null) {
            info = new RT2DocInfo (hzInstance, docUID.getValue());
        }

        return info;
	}

	//-------------------------------------------------------------------------
	/** clean up all resources bound to that document.
	 *
	 *	@param	sDocUID [IN]
	 *			the unique ID for the document where all info object should be removed.
	 */
	public void freeDocInfos (final RT2DocUidType sDocUID) {
	    log.debug("freeDocInfos for docUid [{}]", sDocUID);
		docInfoRegistry.remove(sDocUID);
	}
    
}
