/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.jms;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.apache.activemq.command.ActiveMQQueue;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jms.JmsException;
import org.springframework.stereotype.Service;

import com.google.protobuf.GeneratedMessageV3;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.Member;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.RT2LogInfo;
import com.openexchange.office.rt2.core.exception.RT2TypedException;
import com.openexchange.office.rt2.core.metric.DocProcessorMetricService;
import com.openexchange.office.rt2.core.metric.DocProxyRequestMetricService;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.rt2.protocol.GpbMessageJmsPostProcessor;
import com.openexchange.office.rt2.protocol.RT2GoogleProtocol.BroadcastMessage;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2MessageJmsPostProcessor;
import com.openexchange.office.rt2.protocol.value.RT2AdminIdType;
import com.openexchange.office.rt2.protocol.value.RT2MessageType;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.jms.JmsTemplateWithTtl;
import com.openexchange.office.tools.jms.JmsTemplateWithoutTtl;

@Service
public class RT2JmsMessageSender {

    private static final Logger log = LoggerFactory.getLogger(RT2JmsMessageSender.class);

	//-------------------------------Services------------------------------------------
    @Autowired
    private RT2DocOnNodeMap rt2DocOnNodeMap;

    @Autowired
    private DocProxyRequestMetricService docProxyRequestMetricService;

    @Autowired
    private DocProcessorMetricService docProcessorMetricService;

    @Autowired
    private JmsTemplateWithoutTtl jmsTemplateWithoutTtl;

    @Autowired
    private JmsTemplateWithTtl jmsTemplateWithTtl;

    @Autowired
    private HazelcastInstance hazelcastInstance;

    public void sendBroadcastMessage(BroadcastMessage msg) {
    	ByteArrayOutputStream baOut = new ByteArrayOutputStream();
        try {
        	msg.writeTo(baOut);
        	jmsTemplateWithoutTtl.convertAndSend(RT2JmsDestination.clientResponseTopic, baOut.toByteArray(), new GpbMessageJmsPostProcessor(msg));
        } catch (JmsException | IOException jmsEx) {
        	log.error("sendToClientResponseTopic-Exception", jmsEx);
        }
    }

    public void sendToClientResponseTopic(RT2Message msg) {

        LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, msg.getType().getValue());
        log.debug("Send message to CLIENT_RESPONSE_TOPIC_NAME: {}", msg.getType());
        String rt2MsgAsStr = msg.getBodyString();
        docProcessorMetricService.stopTimer(msg.getMessageID());
        try {
        	jmsTemplateWithoutTtl.convertAndSend(RT2JmsDestination.clientResponseTopic, rt2MsgAsStr, new RT2MessageJmsPostProcessor(msg));
        } catch (JmsException jmsEx) {
        	log.error("sendToClientResponseTopic-Exception", jmsEx);
        }
    }

    public void sendAdminMessage(GeneratedMessageV3 msg) {
    	ByteArrayOutputStream baOut = new ByteArrayOutputStream();
        try {
        	msg.writeTo(baOut);
        	jmsTemplateWithoutTtl.convertAndSend(RT2JmsDestination.adminTopic, baOut.toByteArray(), new GpbMessageJmsPostProcessor());
        } catch (JmsException | IOException jmsEx) {
        	log.error("sendAdminMessage-Exception", jmsEx);
        }
    }

    public void sendToAdminTopic(RT2Message msg, RT2AdminIdType adminId) {
        msg.setAdminID(adminId);
        LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, msg.getType().getValue());
        log.debug("admin send request ''{}''...", msg);
        String rt2MsgAsStr = msg.getBodyString();
        try {
        	jmsTemplateWithoutTtl.convertAndSend(RT2JmsDestination.adminTopic, rt2MsgAsStr, new RT2MessageJmsPostProcessor(msg));
        } catch (JmsException jmsEx) {
        	log.error("sendToAdminTopic-Exception", jmsEx);
        }
    }

    public void sendToDocumentQueue(RT2Message msg, List<RT2LogInfo> msgs) throws RT2TypedException {
        // Cannot be null, because it will be added before
        String docUidAsStr = msg.getDocUID().getValue();
        String nodeId = rt2DocOnNodeMap.get(docUidAsStr);

        if (nodeId == null) {
            // Special case: It's possible that we receive an ack although we
            // already cleaned up client data and sent the leave-response. The
            // client can send it before it receives the leave-response and
            // closes the connection and ends the ACK process. Therefore we just
            // have to ignore ACK_SIMPLE messages in case we cannot find the nodeId.
            if ((msg.getType() == RT2MessageType.ACK_SIMPLE) || msg.getType().equals(RT2MessageType.REQUEST_UNAVAILABILITY)) {
                return;
            }
            log.warn("There has to be an entry in the docOnNodeMap for doc-uid {}, we assume that it was already disposed!", msg.getDocUID().getValue());
            throw new RT2TypedException(ErrorCode.GENERAL_DOCUMENT_ALREADY_DISPOSED_ERROR, new ArrayList<>(msgs));
        }

        if ((msg.getType() == RT2MessageType.REQUEST_JOIN) && !isActiveHzMember(nodeId)) {
            // Special case: It's possible that the DocOnNode map contains invalid node
            // information. It makes no sense to send this join message to a queue which
            // has no consumer. Therefore we bail out here with an exception to provide
            // the client with better information. Hopefully the RT2 GC or the node crashed
            // code will clean-up the DocOnNode map.
            log.warn("Detected that docOnNodeMap entry uuid {} for doc-uid {} is not member of the hz cluster. Clean-up process must reset entry.", nodeId, msg.getDocUID().getValue());
            throw new RT2TypedException(ErrorCode.GENERAL_NODE_IN_MAINTENANCE_MODE_ERROR, new ArrayList<>(msgs));
        }

        msg.setInternalBackendProcessingNode(nodeId);
        String queueName = RT2JmsProperties.DOCUMENT_QUEUE_NAME + nodeId;
        LogProperties.putProperty(LogProperties.Name.RT2_REQUEST_TYPE, msg.getType().getValue());
        log.debug("Sending msg [{}] to documentQueue: [{}]", msg, queueName);
        ActiveMQQueue activeMQQueue = new ActiveMQQueue(queueName);
        String rt2MsgAsStr = msg.getBodyString();
        docProxyRequestMetricService.stopTimer(msg.getMessageID());
        try {
        	jmsTemplateWithTtl.convertAndSend(activeMQQueue, rt2MsgAsStr, new RT2MessageJmsPostProcessor(msg));
        } catch (JmsException jmsEx) {
        	log.error("sendToDocumentQueue-Exception", jmsEx);
        	throw new RT2TypedException(ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR, new ArrayList<>(msgs));
        }
    }

    //-------------------------------------------------------------------------
    private boolean isActiveHzMember(String nodeUuid) {
        if (StringUtils.isEmpty(nodeUuid))
            return false;

        boolean isActive = false;
        final Set<Member> members = hazelcastInstance.getCluster().getMembers();
        if (members != null) {
            isActive = members.stream().anyMatch(m -> nodeUuid.equals(m.getUuid()));
        }
        return isActive;
    }

}
