package com.openexchange.office.rt2.core.jms;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.command.ActiveMQTextMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.jms.listener.DefaultMessageListenerContainer;
import org.springframework.stereotype.Service;
import org.springframework.util.backoff.ExponentialBackOff;

import com.codahale.metrics.Counter;
import com.codahale.metrics.MetricRegistry;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.core.cache.RT2HazelcastHelperService;
import com.openexchange.office.tools.annotation.ShutdownOrder;
import com.openexchange.office.tools.common.jms.JmsMessageListener;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;
import com.openexchange.office.tools.jms.PooledConnectionFactoryProxy;

@Service
@ShutdownOrder(value=-5)
public class RT2DcsUpdateConsumer implements MessageListener, JmsMessageListener, DisposableBean {

	private static final Logger log = LoggerFactory.getLogger(RT2DcsUpdateConsumer.class);

	//--------------------------Services--------------------------------------	
	
    @Autowired
    private PooledConnectionFactoryProxy pooledConnectionFactoryProxy;    

    @Autowired
    private MetricRegistry metricRegistry;
    
    @Autowired
    private RT2HazelcastHelperService rt2HazelcastHelperService;
	
	//------------------------------------------------------------------------	
    private DefaultMessageListenerContainer msgListenerCont;
    
    private Counter counter;
    
    //-------------------------------------------------------------------------
	@Override
	public void onMessage(Message msg) {
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "RT2DcsUpdateConsumer");
        LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_UID, rt2HazelcastHelperService.getHazelcastLocalNodeUuid());      	        
		counter.inc();
		ActiveMQTextMessage txtMsg = (ActiveMQTextMessage) msg;
		try {
			String newBrokerUrl = txtMsg.getText();
			ActiveMQConnectionFactory connFactory = (ActiveMQConnectionFactory) pooledConnectionFactoryProxy.getPooledConnectionFactory().getConnectionFactory();
			if (!newBrokerUrl.equals(connFactory.getBrokerURL())) {
				log.info("Received new brokerURl: {}", newBrokerUrl);			
				connFactory.setBrokerURL(newBrokerUrl);
			}
		} catch (JMSException ex) {
			log.error(ex.getErrorCode(), ex);
		}
	}
	
	//-------------------------------------------------------------------------	
	@Override
	public void startReceiveMessages() {
        if (msgListenerCont == null) {
        	this.counter = metricRegistry.counter(MetricRegistry.name("RT2DcsUpdateConsumer", "count"));
        	
            msgListenerCont = new DefaultMessageListenerContainer();
            ExponentialBackOff exponentialBackOff = new ExponentialBackOff();
            exponentialBackOff.setMaxInterval(60000);
            msgListenerCont.setBackOff(exponentialBackOff);            
            msgListenerCont.setConnectionFactory(pooledConnectionFactoryProxy.getPooledConnectionFactory());
            msgListenerCont.setConcurrentConsumers(3);
            msgListenerCont.setDestination(RT2JmsDestination.brokerChangedTopic);
            msgListenerCont.setMaxConcurrentConsumers(1);
            msgListenerCont.setPubSubDomain(true);
            msgListenerCont.setAutoStartup(true);
            msgListenerCont.setupMessageListener(this);
            msgListenerCont.setTaskExecutor(new SimpleAsyncTaskExecutor(new ThreadFactoryBuilder("RT2DcsUpdateConsumer-%d").build()));
            msgListenerCont.afterPropertiesSet();            
            msgListenerCont.start();
        }
    }	
    
	//-------------------------------------------------------------------------	
    @Override
	public void destroy() throws Exception {
        if (msgListenerCont != null) {
            msgListenerCont.destroy();
            msgListenerCont = null;
        }
	}    
}
