/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.openexchange.office.rt2.core.RT2MessageLoggerService;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorLogInfo.Direction;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.RT2Protocol;

//=============================================================================
@Service
public class RT2MessageTypeDispatcher implements InitializingBean {
    //-------------------------------------------------------------------------
	private static final Logger log = LoggerFactory.getLogger(RT2MessageTypeDispatcher.class);

    //-------------------------------------------------------------------------
	private static final Map<String, String> PREDEFINED_DEFAULT_METHODS_MAP;
	static
	{
		// predefined message types to be mapped to defined methods
		final HashMap<String, String> tmp = new HashMap<>();
        tmp.put(RT2Protocol.REQUEST_JOIN,               "onJoin"                 );
        tmp.put(RT2Protocol.REQUEST_LEAVE,              "onLeave"                );
		tmp.put(RT2Protocol.REQUEST_OPEN_DOC,           "onOpenDoc"              );
		tmp.put(RT2Protocol.REQUEST_CLOSE_DOC,          "onCloseDoc"             );
        tmp.put(RT2Protocol.REQUEST_SWITCH_DOCSTATE,    "onSwitchDocState"       );
		tmp.put(RT2Protocol.REQUEST_APPLY_OPS,          "onApplyOperations"      );
		tmp.put(RT2Protocol.REQUEST_SAVE_DOC,           "onSaveDoc"              );
		tmp.put(RT2Protocol.REQUEST_SYNC,               "onSync"                 );
		tmp.put(RT2Protocol.REQUEST_SYNC_STABLE,        "onSyncStable"           );
		tmp.put(RT2Protocol.REQUEST_EDITRIGHTS,         "onEditRights"           );
		tmp.put(RT2Protocol.REQUEST_UNAVAILABILITY,     "onUnavailable"          );
		tmp.put(RT2Protocol.REQUEST_EMERGENCY_LEAVE,    "onEmergencyLeave"       );

		PREDEFINED_DEFAULT_METHODS_MAP = Collections.unmodifiableMap(tmp);
	}

	private Map<Class<?>, MessageDispatcher> customMsgDispatchers = new HashMap<>();
	
	private Map<Class<?>, MessageDispatcher> defaultMsgDispatchers = new HashMap<>();

	private ReadWriteLock defaultRwLock = new ReentrantReadWriteLock();
	
	private Lock defaultReadLock;
	
	private Lock defaultWriteLock;
	
	private ReadWriteLock customRwLock = new ReentrantReadWriteLock();
	
	private Lock customReadLock;
	
	private Lock customWriteLock;

	@Autowired
    private RT2MessageLoggerService messageLoggerService;

	//-------------------------------------------------------------------------
    @Override
	public void afterPropertiesSet() {
    	defaultReadLock = defaultRwLock.readLock();
    	defaultWriteLock = defaultRwLock.writeLock();
    	customReadLock = customRwLock.readLock();
    	customWriteLock = customRwLock.writeLock();
	}
		
    //-------------------------------------------------------------------------
	public void handleRequest(final RT2Message aRawRequest, RT2DocProcessor docProcessor) throws Exception
	{
		messageLoggerService.addMessageForQueueToLog(Direction.QUEUE, aRawRequest);
		
        boolean bProcessed = processMessage(aRawRequest, docProcessor);
        if (!bProcessed) {
          	log.warn("Message of type {} not successfully processed.", aRawRequest.getType());
        }
	}
	
    //-------------------------------------------------------------------------
	protected boolean processMessage(final RT2Message aMsg, RT2DocProcessor docProcessor) throws Exception
	{
		final String sMsgType = aMsg.getType().getValue();

		// Dispatch message to the implementation method
		// using the default msg dispatcher or a custom msg
		// dispatcher.
		boolean bProcessed = false;
		if (PREDEFINED_DEFAULT_METHODS_MAP.containsKey(sMsgType)) {
			bProcessed = getDefaultMsgDispatcher(docProcessor).callMethod(docProcessor, PREDEFINED_DEFAULT_METHODS_MAP.get(sMsgType), aMsg);
		} else {
			bProcessed = getCustomMsgDispatcher(docProcessor).callMethod(docProcessor, aMsg);
		}
		return bProcessed;
	}
	
	//-------------------------------------------------------------------------	
	private MessageDispatcher getCustomMsgDispatcher(RT2DocProcessor docProcessor) throws Exception {
		customReadLock.lock();
		try {
			MessageDispatcher res = customMsgDispatchers.get(docProcessor.getClass());
			if (res != null) {
				return res;
			}
		} finally {
			customReadLock.unlock();
		}
		customWriteLock.lock();
		try {
			MessageDispatcher res = customMsgDispatchers.get(docProcessor.getClass());
			if (res != null) {
				return res;
			}
			res = new MessageDispatcher(docProcessor.getClass(), "");
			customMsgDispatchers.put(docProcessor.getClass(), res);
			return res;
		} finally {
			customWriteLock.unlock();
		}		
	}
	
	//-------------------------------------------------------------------------	
	private MessageDispatcher getDefaultMsgDispatcher(RT2DocProcessor docProcessor) throws Exception {
		defaultReadLock.lock();
		try {
			MessageDispatcher res = defaultMsgDispatchers.get(docProcessor.getClass());
			if (res != null) {
				return res;
			}
		} finally {
			defaultReadLock.unlock();
		}
		defaultWriteLock.lock();
		try {
			MessageDispatcher res = defaultMsgDispatchers.get(docProcessor.getClass());
			if (res != null) {
				return res;
			}
			res = new MessageDispatcher(docProcessor.getClass(), PREDEFINED_DEFAULT_METHODS_MAP.values());
			defaultMsgDispatchers.put(docProcessor.getClass(), res); 
			return res;
		} finally {
			defaultWriteLock.unlock();
		}				
	}	
}
