package com.openexchange.office.rt2.core.doc;

import java.time.LocalDateTime;

import com.openexchange.office.rt2.core.RT2LogInfo;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2ErrorCodeType;
import com.openexchange.office.rt2.protocol.value.RT2SeqNumberType;
import com.openexchange.office.tools.common.error.ErrorCode;

public class RT2DocProcessorLogInfo implements Comparable<RT2DocProcessorLogInfo>, RT2LogInfo {
	
	public enum Direction {QUEUE, FROM_JMS, TO_JMS, NEW_INSTANCE};
	
	private final Direction direction;
	private final String msgType;
	private final RT2SeqNumberType seqNr;
	private final RT2CliendUidType clientUid;
	private final RT2DocUidType docUid;
	private final RT2ErrorCodeType errorCode;
	private final LocalDateTime occured;
	
	public RT2DocProcessorLogInfo(Direction direction, String msgType, RT2CliendUidType clientUid, RT2DocUidType docUid) {
		this.direction = direction;
		this.msgType = msgType;
		seqNr = new RT2SeqNumberType(-1);
		if (clientUid != null) {
			this.clientUid = clientUid;
		} else {
			this.clientUid = new RT2CliendUidType("not set");
		}
		this.docUid = docUid;
		this.errorCode = new RT2ErrorCodeType(ErrorCode.NO_ERROR);
		this.occured = LocalDateTime.now();
	}
	
	public RT2DocProcessorLogInfo(Direction direction, RT2Message msg) {
		this.direction = direction;
		msgType = msg.getType().getValue();
		if (msg.isSequenceMessage()) {
			seqNr = msg.getSeqNumber();
		} else {
			seqNr = new RT2SeqNumberType(-1);
		}
		clientUid = msg.getClientUID();
		docUid = msg.getDocUID();
		if (msg.isError()) {
			errorCode = msg.getError();
		} else {
			errorCode = new RT2ErrorCodeType(ErrorCode.NO_ERROR);
		}
		this.occured = LocalDateTime.now();
	}

	@Override
	public RT2CliendUidType getClientUid() {
		return clientUid;
	}

	@Override
	public RT2DocUidType getDocUid() {
		return docUid;
	}
	
	@Override
	public boolean isDocProcessorLogInfo() {
		return true;
	}

	@Override
	public int compareTo(RT2DocProcessorLogInfo o) {
		return this.occured.compareTo(o.occured);
	}

	@Override
	public String toString() {
		return "RT2DocProxyLogInfo [direction=" + direction + ", msgType=" + msgType + ", seqNr=" + seqNr
				+ ", clientUid=" + clientUid + ", docUid=" + docUid + ", errorCode=" + errorCode + ", occured="
				+ occured + "]";
	}

}
