/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import org.springframework.beans.factory.annotation.Autowired;

import com.openexchange.config.ConfigurationService;
import com.openexchange.office.document.api.Document;
import com.openexchange.office.rt2.core.RT2MessageLoggerService;
import com.openexchange.office.rt2.core.doc.RT2DocProcessorLogInfo.Direction;
import com.openexchange.office.rt2.core.logging.IMessagesLoggable;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.rt2.protocol.value.RT2SessionIdType;
import com.openexchange.office.tools.common.user.UserData;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.office.tools.service.session.SessionService;

//=============================================================================
/** implements the back bone of a RT2 document ...
 *  This instance exists one times for every document within the cluster.
 *  It knows all states about those document and control it's life cycle.
 */
public abstract class RT2DocProcessor implements Document, IMessagesLoggable {

	public enum ProcessingState {
		CREATED,
		INITIALIZING,
		OPEN,
		DISPOSED
	}	
	
    private RT2DocUidType m_sDocUID = null;
    
    protected AtomicReference<ProcessingState> m_aProcState = new AtomicReference<>(ProcessingState.CREATED);
   
    //----------------------------Services-------------------------------------
    @Autowired
    protected RT2MessageLoggerService rt2MessageLoggerService;
    
    @Autowired
    protected RT2DocStateNotifier rt2DocStateNotifier;
    
    @Autowired
    protected SessionService sessionService;
    
    @Autowired
    protected ConfigurationService configurationService;
    
    @Autowired
    protected Statistics statistics;
    
    @Autowired
    protected DocumentEventHelper documentEventHelper;
    
    //-------------------------------------------------------------------------
	public abstract String getDocTypeIdentifier();

    //-------------------------------------------------------------------------
	public abstract List<DocProcessorClientInfo> getClientsInfo();
	
	//-------------------------------------------------------------------------
	public abstract void updateClientsInfo(Collection<RT2CliendUidType> existingClients);

	//-------------------------------------------------------------------------	
	public abstract boolean enqueueMessage(final RT2Message msg) throws Exception;
	
	//=============================================================================
	// Default methods for predefined messages
	//=============================================================================

    //-------------------------------------------------------------------------
    public abstract boolean onJoin(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
    public abstract boolean onLeave(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onAdditionalOpenDoc(final RT2Message aMsg) throws Exception;    

    //-------------------------------------------------------------------------
	public abstract boolean onFirstOpenDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onOpenDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onCloseDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onApplyOperations(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onSaveDoc(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onSync(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onSyncStable(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onEditRights(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onUnavailable(final RT2Message aMsg) throws Exception;

    //-------------------------------------------------------------------------
	public abstract boolean onEmergencyLeave(final RT2Message aMsg) throws Exception;

	public abstract List<UserData> getUserDatasCloned();

	public abstract int getPendingOperationsCount();
	
	public void init() throws Exception {};
	
	//-------------------------------------------------------------------------
	public synchronized void setDocUID (final RT2DocUidType docUID) {
		m_sDocUID = docUID;
		rt2MessageLoggerService.add(docUID, Direction.NEW_INSTANCE, "RT2DocProcessor::constructor", null);
		rt2DocStateNotifier.notifyEvent(RT2DocStateNotifier.DOCSTATE_EVENT_CREATED, this);
	}
	
	//-------------------------------------------------------------------------	
	public Map<RT2SessionIdType, RT2CliendUidType> getSessionIdToClientUids() {
		Map<RT2SessionIdType, RT2CliendUidType> res = new HashMap<>();
		for (UserData userData : getUserDatasCloned()) {
			res.put(new RT2SessionIdType(userData.getSessionId()), new RT2CliendUidType(userData.getClientUid()));
		}
		return res;
	}

	//-------------------------------------------------------------------------
	public boolean isDisposed() {
        return (m_aProcState.get() == ProcessingState.DISPOSED);
    }

    //-------------------------------------------------------------------------
	protected void setDisposed() {
        m_aProcState.set(ProcessingState.DISPOSED);
	}
		
    //-------------------------------------------------------------------------
	public void dispose() {
		rt2DocStateNotifier.notifyEvent(RT2DocStateNotifier.DOCSTATE_EVENT_DISPOSED, this);
    }

    //-------------------------------------------------------------------------
	public List<String> formatMsgsLogInfoForClient(RT2CliendUidType clientUid) {
    	return rt2MessageLoggerService.formatMsgsLogInfoForClient(getDocUID(), clientUid);
	}

    //-------------------------------------------------------------------------
    @Override
	public RT2DocUidType getDocUID () {
		return m_sDocUID;
	}

    //-------------------------------------------------------------------------
    @Override
	public List<String> formatMsgsLogInfo() {
    	return rt2MessageLoggerService.formatMsgsLogInfo(getDocUID());
	}

    //-------------------------------------------------------------------------
	@Override
	public RT2CliendUidType getClientUID() {
		return null;
	}

	//-------------------------------------------------------------------------	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((m_sDocUID == null) ? 0 : m_sDocUID.hashCode());
		return result;
	}

    //-------------------------------------------------------------------------	
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		RT2DocProcessor other = (RT2DocProcessor) obj;
		if (m_sDocUID == null) {
			if (other.m_sDocUID != null)
				return false;
		} else if (!m_sDocUID.equals(other.m_sDocUID))
			return false;
		return true;
	}
}
