/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rt2.core.doc;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DocStatistics implements IDocNotificationHandler, IStatisticData
{
    //-------------------------------------------------------------------------
	private static final Logger log = LoggerFactory.getLogger(DocStatistics.class);

    //-------------------------------------------------------------------------
    private static class Holder
    {
        static final DocStatistics INSTANCE = new DocStatistics();
    }

    //-------------------------------------------------------------------------
    private AtomicInteger           m_aNum              = new AtomicInteger(0);
    private Map<String, MutableInt> m_aNumForDocTypeMap = new HashMap<>();

    //-------------------------------------------------------------------------
    private DocStatistics()
    {
    }

    //-------------------------------------------------------------------------
    @Override
    public void docProcessorCreated(final RT2DocProcessor aCreatedInstance)
    {
        m_aNum.incrementAndGet();

        final String sDocType = aCreatedInstance.getDocTypeIdentifier();
        synchronized(m_aNumForDocTypeMap)
        {
            MutableInt aInteger = m_aNumForDocTypeMap.get(sDocType);
            if (aInteger == null)
            {
                aInteger = new MutableInt(1);
                m_aNumForDocTypeMap.put(sDocType, aInteger);
            }
            else
                aInteger.increment();
        }
    }

    //-------------------------------------------------------------------------
    @Override
    public void docProcessorDisposed(final RT2DocProcessor aDisposedInstance)
    {
        m_aNum.decrementAndGet();

        final String sDocType = aDisposedInstance.getDocTypeIdentifier();
        synchronized(m_aNumForDocTypeMap)
        {
            MutableInt aInteger = m_aNumForDocTypeMap.get(sDocType);
            if (aInteger == null)
            {
                log.warn("RT2StatisticDocManager: docProcessorDisposed received but no number instance available");
            }
            else
                aInteger.decrement();
        }
    }

	//-------------------------------------------------------------------------
    public int getNumberOfDocs()
    {
        return m_aNum.get();
    }

    //-------------------------------------------------------------------------
    public DocStatistics getInstance()
    {
        return Holder.INSTANCE;
    }

    //-------------------------------------------------------------------------
    @Override
    public int getNumberOfOpenDocuments()
    {
        return m_aNum.get();
    }

    //-------------------------------------------------------------------------
    @Override
    public int getNumberOfOpenDocuments(String sDocType)
    {
        synchronized (m_aNumForDocTypeMap)
        {
            final MutableInt aInteger = m_aNumForDocTypeMap.get(sDocType);
            return (aInteger != null) ? aInteger.getValue() : 0;
        }
    }

}
