/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks of the OX Software GmbH group of companies.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.rt2.core.control.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;

import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.rt2.core.control.Task;
import com.openexchange.office.rt2.core.proxy.RT2DocProxy;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyRegistry;
import com.openexchange.office.rt2.hazelcast.RT2DocOnNodeMap;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.tools.common.log.LogMethodCallHelper;
import com.openexchange.office.tools.common.log.Loggable;

@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class CleanupTask extends Task implements Loggable
{
	private static final Logger log = LoggerFactory.getLogger(CleanupTask.class);
	
    //-------------------------------------------------------------------------
    private final String  m_sMasterUUID; 
    
	private final String  m_sNodeUUIDToCleanup;

    private AtomicBoolean m_bSuccessful = new AtomicBoolean(false);	
	
	//-------------------------------------------------------------------------
	@Autowired
	private RT2DocProxyRegistry rt2DocProxyRegistry;
	
	@Autowired
	private RT2DocOnNodeMap docOnNodeMap;

    //-------------------------------------------------------------------------
    public CleanupTask(String sTaskID, String sMasterUUID, String sNodeUUIDToCleanup)
    {
        super(sTaskID);
        m_sMasterUUID        = sMasterUUID;
        m_sNodeUUIDToCleanup = sNodeUUIDToCleanup;
    }

    //-------------------------------------------------------------------------
    public String getMasterUUID()
    {
        return m_sMasterUUID;
    }

    //-------------------------------------------------------------------------
    public String getMemberUUIDToCleanup()
    {
        return m_sNodeUUIDToCleanup;
    }

    //-------------------------------------------------------------------------
    public boolean successful()
    {
        return m_bSuccessful.get();
    }

    //-------------------------------------------------------------------------
    public boolean failed()
    {
        return !successful();
    }

    //-------------------------------------------------------------------------
    @Override
    public boolean process() throws Exception {
    	LogMethodCallHelper.logMethodCall(this, "process");
        try {
            final Set<String>     aDocUIDs      = docOnNodeMap.getDocsOfMember(m_sNodeUUIDToCleanup);

            if (!aDocUIDs.isEmpty()) {
                int nCrashedDocsToNotify = aDocUIDs.size();
                log.info("RT2: CleanupTask found count documents {} controlled by crashed node with uid {}", nCrashedDocsToNotify, m_sNodeUUIDToCleanup);
                for (final String sDocUID : aDocUIDs)
                    notifyAndCleanupDocProxiesOfDocument(sDocUID);
            } else {
                log.info("RT2: CleanupTask found no document controlled by crashed node with uid {}", m_sNodeUUIDToCleanup);
            }
            m_bSuccessful.set(true);
        }
        finally {
            setCompleted(true);
        }
        return true;
    }

    //-------------------------------------------------------------------------
    @Override
    public String toString()
    {
        final StringBuilder aTmp = new StringBuilder(super.toString());
        aTmp.append(",member uuid=");
        aTmp.append(m_sMasterUUID);
        aTmp.append(",master uuid=");
        aTmp.append(this.getMasterUUID());
        aTmp.append(",cleanup member uuid=");
        aTmp.append(this.getMemberUUIDToCleanup());
        return aTmp.toString();
    }

    //-------------------------------------------------------------------------
    private void notifyAndCleanupDocProxiesOfDocument(final String sDocUID) throws Exception
    {
        final List<RT2DocProxy> aProxies     = rt2DocProxyRegistry.getDocProxies4DocUID(new RT2DocUidType(sDocUID), null);

        if (!aProxies.isEmpty()) {
            for (final RT2DocProxy aDocProxy : aProxies) {
                try {
                    log.debug("RT2: Try to send 'crashed' broadcast and cleanup for docUid {}", aDocProxy.getDocUID());
                    aDocProxy.sendCrashedResponseToClient();
                    cleanupDocProxyResources(aDocProxy);
                }
                catch (Throwable t) {
                    ExceptionUtils.handleThrowable(t);
                    log.error("RT2: CleanupTask caught exception on notifying/cleanup document proxy with docUid  + aDocProxy.getDocUID()", t);
                }
            }
        }
    }

    //-------------------------------------------------------------------------
    private void cleanupDocProxyResources(final RT2DocProxy aDocProxy) throws Exception
    {
    	rt2DocProxyRegistry.deregisterDocProxy(aDocProxy, true);
        aDocProxy.forceCleanup();
    }

	@Override
	public Logger getLogger() {
		return log;
	}

	@Override
	public Map<String, Object> getAdditionalLogInfo() {
		Map<String, Object> res = new HashMap<>();
		res.put("NodeUUIDToCleanup", m_sNodeUUIDToCleanup);
		return res;
	}

}
