/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest.tools;

import java.util.Arrays;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.office.tools.service.files.FileHelperService;
import com.openexchange.office.tools.service.files.FileHelperServiceFactory;
import com.openexchange.office.tools.service.session.SessionService;
import com.openexchange.tools.session.ServerSession;

@Service
public class ParamValidatorService {

    protected static final Logger log = LoggerFactory.getLogger(ParamValidatorService.class);

    @Autowired
    private SessionService sessionService;
    
    @Autowired
    private FileHelperServiceFactory fileHelperServiceRegistry;
    
    // ---------------------------------------------------------------
    public boolean areAllParamsNonEmpty(final AJAXRequestData aRequest, String[] paramNames) {

        return (aRequest != null) &&
            Arrays.asList(paramNames)
                  .stream()
                  .map(p -> aRequest.getParameter(p))
                  .filter(StringUtils::isEmpty)
                  .collect(Collectors.toSet())
                  .isEmpty();
    }

    // ---------------------------------------------------------------
    public boolean canReadFile(ServerSession aSession, String folderId, String fileId) {
        boolean canRead = false;
        try {
            final int userId = sessionService.getUserId(aSession);

            // check document access rights to ensure that the current user has access rights to retrieve document content
            final FileHelperService fileHelperService = fileHelperServiceRegistry.create(aSession);
            canRead = fileHelperService.canReadFile(folderId, fileId, userId);
        } catch (Exception e) {
            log.error("AjaxRequest GetFileAction exception caught while trying to check access rights", e);
        }

        return canRead;
    }

    // ---------------------------------------------------------------
    public static <T> T getParameterOrDefault(final AJAXRequestData request, String paramName, Class<T> clazz, T defaultValue) throws OXException {
    	T value = request.getParameter(paramName, clazz, true);
    	if (value == null) {
    		value = defaultValue;
    	}

    	return value;
    }

    // ---------------------------------------------------------------
    public static AJAXRequestResult getResultFor(int nHttpError) {
        final AJAXRequestResult result = new AJAXRequestResult();
        result.setHttpStatusCode(nHttpError);
        return result;
    }
}
