/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.docx.tools;

import java.util.Set;
import org.docx4j.wml.P;
import org.docx4j.wml.R;
import org.docx4j.wml.RDBase;
import org.docx4j.wml.RIBase;
import org.docx4j.wml.RunDel;
import org.docx4j.wml.RunIns;
import org.docx4j.wml.RunMoveFrom;
import org.docx4j.wml.RunMoveTo;
import org.docx4j.wml.SdtRun;
import com.google.common.collect.ImmutableSet;
import com.openexchange.office.filter.core.IContentAccessor;
import com.openexchange.office.filter.core.component.Child;

public class TextRunState implements ITextRunState {

    public P paragraph = null;
    public R textRun = null;
    public P.Hyperlink hyperlink = null;
    public RunMoveTo runMoveTo = null;
    public RunMoveFrom runMoveFrom = null;
    public RunIns runIns = null;
    public RunDel runDel = null;

    public TextRunState(R textRun) {
        this.textRun = textRun;
        Child parent = textRun;
        do {
            parent = (Child)parent.getParent();
            if(parent instanceof P.Hyperlink) {
                hyperlink = (P.Hyperlink)parent;
            }
            else if(parent instanceof RunMoveTo) {
                runMoveTo = (RunMoveTo)parent;
            }
            else if(parent instanceof RunMoveFrom) {
                runMoveFrom = (RunMoveFrom)parent;
            }
            else if(parent instanceof RunIns) {
                runIns = (RunIns)parent;
            }
            else if(parent instanceof RunDel) {
                runDel = (RunDel)parent;
            }
        }
        while(!(parent instanceof P));
        paragraph = (P)parent;
    }

    @Override
    public P getParagraph() {
        return paragraph;
    }

    @Override
    public P.Hyperlink getHyperlink() {
        return hyperlink;
    }

    @Override
    public R getTextRun() {
        return textRun;
    }

    @Override
    public boolean hasChangeTracking() {
        return runIns!=null||runDel!=null||runMoveFrom!=null||runMoveTo!=null;
    }

    @Override
    public P.Hyperlink getHyperlink(boolean forceCreate) {
        if(hyperlink==null&&forceCreate) {
            hyperlink = new P.Hyperlink();
            insertChild(hyperlink, hyperlinkParentContextList);
        }
        return hyperlink;
    }

    @Override
    public void removeHyperlink() {
        if(hyperlink!=null) {
            deleteChild(hyperlink);
            hyperlink = null;
        }
    }

    static final private ImmutableSet<Class<?>> hyperlinkParentContextList = ImmutableSet.<Class<?>> builder()
        .add(P.class)
        .add(SdtRun.class)
        .build();

    @Override
    public RIBase getRunIns(boolean forceCreate) {
        if(runIns==null&&runMoveTo==null&&forceCreate) {
            runIns = new RunIns();
            insertChild(runIns, runInsParentContextList);
        }
        return runIns!=null ? runIns : runMoveTo;
    }

    @Override
    public void removeRunIns() {
        if(runIns!=null) {
            deleteChild(runIns);
            runIns = null;
        }
        if(runMoveTo!=null) {
            deleteChild(runMoveTo);
            runMoveTo = null;
        }
    }

    static final private ImmutableSet<Class<?>> runInsParentContextList = ImmutableSet.<Class<?>> builder()
        .add(P.class)
        .add(SdtRun.class)
        .add(P.Hyperlink.class)
        .build();

    @Override
    public RDBase getRunDel(boolean forceCreate) {
        if(runDel==null&&runMoveFrom==null&&forceCreate) {
            runDel = new RunDel();
            insertChild(runDel, runDelParentContextList);
        }
        return runDel!=null ? runDel : runMoveFrom;
    }

    @Override
    public void removeRunDel() {
        if(runDel!=null) {
            deleteChild(runDel);
            runDel = null;
        }
        if(runMoveFrom!=null) {
            deleteChild(runMoveFrom);
            runMoveFrom = null;
        }
    }

    static final private ImmutableSet<Class<?>> runDelParentContextList = ImmutableSet.<Class<?>> builder()
        .add(P.class)
        .add(SdtRun.class)
        .add(P.Hyperlink.class)
        .add(RunMoveFrom.class)
        .add(RunMoveTo.class)
        .add(RunIns.class)
        .build();

    public void insertChild(Object c, Set<Class<?>> parentContextList) {
        Object child = textRun;
        Object parent = ((Child)child).getParent();
        while(parent!=null&&!parentContextList.contains(parent.getClass())) {
            child = parent;
            parent = ((Child)child).getParent();
        }
        if(parent!=null) {
            final int index = ((IContentAccessor)parent).getContent().indexOf(child);
            ((IContentAccessor)parent).getContent().set(index, c);
            ((IContentAccessor)c).getContent().add(child);
            ((Child)child).setParent(c);
            ((Child)c).setParent(parent);
        }
    }

    public void deleteChild(Object c) {
        final Object parent = ((Child)c).getParent();
        int index = ((IContentAccessor)parent).getContent().indexOf(c);
        ((IContentAccessor)parent).getContent().remove(index);
        for(Object o:((IContentAccessor)c).getContent()) {
            if(o instanceof Child) {
                ((Child)o).setParent(parent);
            }
            ((IContentAccessor)parent).getContent().add(index++, o);
        }
    }
}
