/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.odt.dom;

import java.util.Map;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.INodeAccessor;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odf.draw.Transformer;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleFamily;
import com.openexchange.office.filter.odf.styles.StyleManager;

public class Annotation implements IElementWriter, INodeAccessor {

	private final AttributesImpl attributes;
    protected final Transformer transformer;
	private DLList<Object> content;
	private String id;
	private String creator;
	private String initials;
	private String date;
	private boolean modifiedTransformer = false;

	// if and only if the annotation was added into the authors list (odp) then
	// this member contains the id that is used in the authors list.
	private Integer authorId;

	public Annotation(String id, Attributes attributes, boolean useTransformation) {
		this.attributes = new AttributesImpl(attributes);
		this.attributes.remove("office:name");
		this.id = id;
        this.transformer = useTransformation ? new Transformer(this.attributes, false) : null;
	}

	public Annotation(String id, boolean useTransformation) {
		this.attributes = new AttributesImpl();
		this.id = id;
		this.transformer = useTransformation ? new Transformer(false) : null;
	}

	public AttributesImpl getAttributes() {
		return attributes;
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getInitials() {
	    return initials;
	}

	public void setInitials(String initials) {
	    this.initials = initials;
	}

	public String getCreator() {
		return creator;
	}

	public void setCreator(String creator) {
		this.creator = creator;
	}

	public String getDate() {
		return date;
	}

	public void setDate(String date) {
		this.date = date;
	}

    public String getStyleName() {
        return attributes.getValue("draw:style-name");
    }

    public void setStyleName(String style) {
        attributes.setValue(Namespaces.DRAW, "style-name", "draw:style-name", style);
    }

    public String getTextStyleName() {
        return attributes.getValue("draw:text-style-name");
    }

    public void setTextStyleName(String name) {
        attributes.setValue(Namespaces.DRAW, "text-style-name", "draw:text-style-name", name);
    }

    public Transformer getTransformer() {
        return transformer;
    }

    public Integer getAuthorId() {
        return authorId;
    }

    public void setAuthorId(int authorId) {
        this.authorId = authorId;
    }

    @Override
	public DLList<Object> getContent() {
        if (content == null) {
            content = new DLList<Object>();
        }
		return content;
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		SaxContextHandler.startElement(output, Namespaces.OFFICE, "annotation", "office:annotation");
		if(id!=null&&!id.isEmpty()) {
			SaxContextHandler.addAttribute(output, Namespaces.OFFICE, "name", "office:name", id);
		}
		if(modifiedTransformer) {
            attributes.setValue(Namespaces.SVG, "x", "svg:x", (transformer.getX() / 100.0 + "mm"));
            attributes.setValue(Namespaces.SVG, "y", "svg:y", (transformer.getY() / 100.0 + "mm"));
            attributes.setValue(Namespaces.SVG, "width", "svg:width", (transformer.getWidth() / 100.0 + "mm"));
            attributes.setValue(Namespaces.SVG, "height", "svg:height", (transformer.getHeight() / 100.0 + "mm"));
	    }
		attributes.write(output);
		if(creator!=null) {
			output.startElement(Namespaces.DC, "creator", "dc:creator");
			output.characters(creator);
			output.endElement(Namespaces.DC, "creator", "dc:creator");
		}
		if(initials!=null) {
		    output.startElement(Namespaces.LOEXT, "sender-initials", "loext:sender-initials");
		    output.characters(initials);
		    output.endElement(Namespaces.LOEXT, "sender-initials", "loext:sender-initials");
		}
		if(date!=null) {
			output.startElement(Namespaces.DC, "date", "dc:date");
			output.characters(date);
			output.endElement(Namespaces.DC, "date", "dc:date");
		}
		TextContentHelper.write(output, getContent());
		SaxContextHandler.endElement(output, Namespaces.OFFICE, "annotation", "office:annotation");
	}

    public void createAttrs(OdfOperationDoc operationDocument, boolean contentAutoStyle, OpAttrs attrs) {
        if(getStyleName()!=null&&!getStyleName().isEmpty()) {
            operationDocument.getDocument().getStyleManager().
                collectAllGraphicPropertiesFromStyle(attrs, getStyleName(), StyleFamily.GRAPHIC, contentAutoStyle, true);
        }
        final String textStyleName = getTextStyleName();
        if(getTextStyleName()!=null&&!getTextStyleName().isEmpty()) {
            operationDocument.getDocument().getStyleManager().createAutoStyleAttributes(attrs, textStyleName, StyleFamily.PARAGRAPH, contentAutoStyle);
        }

        final OpAttrs drawingAttrs = attrs.getMap(OCKey.DRAWING.value(), true);
        drawingAttrs.put(OCKey.HIDDEN.value(), !attributes.getBooleanValue("office:display", false));
        if(transformer!=null) {
            int x = Double.valueOf(transformer.getX()).intValue();
            int y = Double.valueOf(transformer.getY()).intValue();
            int width = Double.valueOf(transformer.getWidth()).intValue();
            int height = Double.valueOf(transformer.getHeight()).intValue();
            drawingAttrs.put(OCKey.ANCHOR.value(), "a");
            drawingAttrs.put(OCKey.WIDTH.value(), width);
            drawingAttrs.put(OCKey.HEIGHT.value(), height);
            drawingAttrs.put(OCKey.LEFT.value(), x);
            drawingAttrs.put(OCKey.TOP.value(), y);

            final Double rotation = transformer.getRotation();
            if(rotation!=null) {
                drawingAttrs.put(OCKey.ROTATION.value(), rotation);
            }
        }
        if(!drawingAttrs.containsKey(OCKey.HEIGHT.value())) {
            final Map<String, Object> shapeAttrs = attrs.getMap(OCKey.SHAPE.value(), true);
            shapeAttrs.put(OCKey.AUTO_RESIZE_HEIGHT.value(), true);
        }
    }

    public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs, boolean contentAutoStyle)
        throws JSONException {

        final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
        final StyleBase styleBase = styleManager.getStyleBaseClone(StyleFamily.GRAPHIC, getStyleName(), contentAutoStyle);
        styleManager.applyAttrs(styleBase, contentAutoStyle, attrs);
        setStyleName(styleManager.getStyleIdForStyleBase(styleBase));
        final StyleBase textStyleBase = styleManager.getStyleBaseClone(StyleFamily.PARAGRAPH, getTextStyleName(), contentAutoStyle);
        styleManager.applyAttrs(textStyleBase, contentAutoStyle, attrs);
        setTextStyleName(styleManager.getStyleIdForStyleBase(textStyleBase));
        final JSONObject drawingProps = attrs.optJSONObject(OCKey.DRAWING.value());
        if(drawingProps!=null) {
            if (drawingProps.has(OCKey.NAME.value())) {
                final String name = drawingProps.optString(OCKey.NAME.value());
                if (name!=null && !name.isEmpty()) {
                    attributes.setValue(Namespaces.DRAW, "name", "draw:name", name);
                }
            }
            Object hidden = drawingProps.opt(OCKey.HIDDEN.value());
            if(hidden!=null) {
                attributes.setBooleanValue(Namespaces.OFFICE, "display", "office:display", !((Boolean)hidden).booleanValue());
            }
            if(transformer!=null) {
                if (drawingProps.has(OCKey.LEFT.value())) {
                    transformer.setX(drawingProps.optInt(OCKey.LEFT.value()));
                }
                if (drawingProps.has(OCKey.TOP.value())) {
                    transformer.setY(drawingProps.optInt(OCKey.TOP.value()));
                }
                if (drawingProps.has(OCKey.WIDTH.value())) {
                    transformer.setWidth(drawingProps.optInt(OCKey.WIDTH.value()));
                }
                if (drawingProps.has(OCKey.HEIGHT.value())) {
                    transformer.setHeight(drawingProps.optInt(OCKey.HEIGHT.value()));
                }
                final Object rotation = drawingProps.opt(OCKey.ROTATION.value());
                if(rotation!=null) {
                    transformer.setRotation(rotation instanceof Number ? ((Number)rotation).doubleValue() : null);
                }
                modifiedTransformer = true;
            }
        }
        final JSONObject shapeProps = attrs.optJSONObject(OCKey.SHAPE.value());
        if(shapeProps!=null) {
            if (shapeProps.has(OCKey.AUTO_RESIZE_HEIGHT.value()) && shapeProps.optBoolean(OCKey.AUTO_RESIZE_HEIGHT.value()) == true) {
                attributes.remove("svg:height");
            }
        }
    }
}
