/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.documents.access.impl;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jms.JmsException;
import org.springframework.stereotype.Service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.office.document.api.Document;
import com.openexchange.office.document.api.DocumentDisposer;
import com.openexchange.office.documents.access.DocRequester;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.doc.OXDocumentException;
import com.openexchange.office.tools.jms.JmsTemplateWithoutTtl;
import com.openexchange.office.tools.jms.OfficeJmsDestination;
import com.openexchange.office.tools.service.caching.HazelcastUuidHelper;
import com.openexchange.office.tools.service.json.ObjectMapperWrapper;

@Service
@RegisteredService(registeredClass=DocRequester.class)
public class DocRequesterImpl implements DocRequester {

    // ---------------------------------------------------------------
    private static final Logger log = LoggerFactory.getLogger(DocRequesterImpl.class);
	
	@Autowired
	private DocumentRequestProcessor documentRequestProcessor;
	
	@Autowired
	private DocumentDisposer documentDisposer;
	
	@Autowired
	private JmsTemplateWithoutTtl jmsTemplate;
	
	@Autowired
	private ObjectMapperWrapper objectMapperWrapper;
	
    @Autowired
    private CorrelationIdDisposer correlationIdDisposer;
	
	//-------------------------------------------------------------------------
	public CompletableFuture<InputStream> sendSyncRequest(RT2DocUidType docUid, RT2CliendUidType clientUid) throws OXException {			
		
		DocRequestData docRequestData = new DocRequestData(docUid.getValue(), clientUid.getValue(), HazelcastUuidHelper.getNodeUuidAsStr());

        log.debug("RT2: sendSyncRequest {}", docRequestData.toString());
        
        ImmutablePair<String, CompletableFuture<InputStream>> p = correlationIdDisposer.createCorrelationId(); 
        String correlationID = p.getKey();
        CompletableFuture<InputStream> result = p.getValue();
        final Document aDoc = getLocalDocumentFor(docUid.getValue());
        if (null == aDoc) {
        	try {
        		jmsTemplate.convertAndSend(OfficeJmsDestination.requestDocTopic, objectMapperWrapper.getObjectMapper().writeValueAsString(docRequestData), (msg) -> {
        			msg.setJMSCorrelationID(correlationID);
        			return msg;
        		});
	        } catch (JmsException | JsonProcessingException jmsEx) {    
	        	log.error("sendSyncRequest-Exception", jmsEx);
	        	throw new OXDocumentException("", ErrorCode.GENERAL_SYSTEM_BUSY_ERROR);
	        }
        } else {
            ManagedFile managedFile = documentRequestProcessor.processDocRequest(docUid, clientUid);
            InputStream isSteam = managedFile.getInputStream(); 
            result.complete(isSteam);
        }
        return result;
    }
	
    //-------------------------------------------------------------------------
	private Document getLocalDocumentFor(String docUID) throws OXDocumentException {
        return documentDisposer.getDocument(docUID);
	}
}
