package com.openexchange.office.documents.access.impl;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.Session;
import javax.jms.TextMessage;

import org.apache.activemq.command.ActiveMQQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.SimpleAsyncTaskExecutor;
import org.springframework.jms.core.MessageCreator;
import org.springframework.jms.listener.DefaultMessageListenerContainer;
import org.springframework.stereotype.Service;
import org.springframework.util.backoff.ExponentialBackOff;

import com.google.common.io.ByteStreams;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.log.LogProperties;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;
import com.openexchange.office.tools.common.jms.JmsMessageListener;
import com.openexchange.office.tools.common.threading.ThreadFactoryBuilder;
import com.openexchange.office.tools.jms.JmsTemplateWithoutTtl;
import com.openexchange.office.tools.jms.OfficeJmsDestination;
import com.openexchange.office.tools.jms.PooledConnectionFactoryProxy;
import com.openexchange.office.tools.service.caching.HazelcastUuidHelper;
import com.openexchange.office.tools.service.json.ObjectMapperWrapper;

@Service
public class DocRequestJmsConsumer implements MessageListener, JmsMessageListener, DisposableBean {

    private static final Logger log = LoggerFactory.getLogger(DocRequestJmsConsumer.class);

    @Autowired
    private PooledConnectionFactoryProxy pooledConnectionFactoryProxy;
    
    @Autowired
    private ObjectMapperWrapper objectMapperWrapper;
    
    @Autowired
    private DocumentRequestProcessor documentRequestProcessor;
    
	@Autowired
	private JmsTemplateWithoutTtl jmsTemplate;
    
	//------------------------------------------------------------------------    
    private DefaultMessageListenerContainer msgListenerCont;

	//------------------------------------------------------------------------
	@Override
	public void startReceiveMessages() {
    	if (msgListenerCont == null) {
            msgListenerCont = new DefaultMessageListenerContainer();
            ExponentialBackOff exponentialBackOff = new ExponentialBackOff();
            exponentialBackOff.setMaxInterval(60000);
            msgListenerCont.setBackOff(exponentialBackOff);            
            msgListenerCont.setConnectionFactory(pooledConnectionFactoryProxy.getPooledConnectionFactory());
            msgListenerCont.setConcurrentConsumers(1);
            msgListenerCont.setDestination(OfficeJmsDestination.requestDocTopic);
            msgListenerCont.setMaxConcurrentConsumers(1);
            msgListenerCont.setPubSubDomain(true);
            msgListenerCont.setAutoStartup(true);
            msgListenerCont.setupMessageListener(this);
            msgListenerCont.setTaskExecutor(new SimpleAsyncTaskExecutor(new ThreadFactoryBuilder("DocRequestJmsConsumer-%d").build()));
            msgListenerCont.afterPropertiesSet();
            msgListenerCont.start();
    	}
	}

	//------------------------------------------------------------------------    
	@Override
	public void destroy() throws Exception {
        if (msgListenerCont != null) {
            msgListenerCont.destroy();
        }		
	}

	//------------------------------------------------------------------------    
    @Override
    public void onMessage(Message jmsMsg) {
    	System.out.println("----------------------------------------DocRequestJmsConsumer");
        try {
        	TextMessage txtJmsMsg = (TextMessage) jmsMsg;
        	DocRequestData docRequestData = objectMapperWrapper.getObjectMapper().readValue(txtJmsMsg.getText(), DocRequestData.class);
        	LogProperties.putProperty(LogProperties.Name.RT2_DOC_UID, docRequestData.getDocUid());
        	LogProperties.putProperty(LogProperties.Name.RT2_CLIENT_UID, docRequestData.getClientUid());
        	LogProperties.putProperty(LogProperties.Name.RT2_BACKEND_PART, "docRequest");
        	try {
	        	ManagedFile managedFile = documentRequestProcessor.processDocRequest(new RT2DocUidType(docRequestData.getDocUid()), new RT2CliendUidType(docRequestData.getClientUid()));
	        	ActiveMQQueue queue = OfficeJmsDestination.getResponseDocQueue(docRequestData.getNodeUid());
	        	jmsTemplate.send(queue, new MessageCreator() {
	        		public Message createMessage(Session session) throws JMSException {
	        			BytesMessage bytesMessage = session.createBytesMessage();
	        			bytesMessage.setJMSCorrelationID(jmsMsg.getJMSCorrelationID());
	        			try {
	        				bytesMessage.writeBytes(ByteStreams.toByteArray(managedFile.getInputStream()));        				
	        			} catch (Exception e) {
	        				log.error(e.getMessage(), e);        				
	        			}
	        			return bytesMessage;
	        	    }
	        	});
        	} catch (DocumentNotFoundException ex) {
        		log.debug("Document with docUid {} not found on node {}", docRequestData.getDocUid(), HazelcastUuidHelper.getNodeUuidAsStr());
        	}
        } catch (Exception e) {
        	log.error(e.getMessage(), e);
        } finally {
            LogProperties.removeLogProperties();
        }
    }
}
