/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.datasource.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.EnumSet;
import java.util.Optional;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import com.openexchange.ajax.requesthandler.crypto.CryptographicServiceAuthenticationFactory;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographicAwareIDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographyMode;
import com.openexchange.office.datasource.access.DataSource;
import com.openexchange.office.datasource.access.DataSourceAccess;
import com.openexchange.office.datasource.access.DataSourceException;
import com.openexchange.office.datasource.access.MetaData;
import com.openexchange.office.tools.common.IOHelper;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.error.ExceptionToErrorCode;
import com.openexchange.office.tools.service.files.FileHelperService;
import com.openexchange.office.tools.service.files.FileHelperServiceFactory;
import com.openexchange.office.tools.service.session.SessionService;
import com.openexchange.tools.session.ServerSession;

public class DataSourceAccessDrive extends DataSourceAccessBase implements DataSourceAccess {
	private static final Logger LOG = LoggerFactory.getLogger(DataSourceAccessDrive.class);

	@Autowired
	private FileHelperServiceFactory fileHelperServiceFactory;
	
	@Autowired
	private SessionService sessionService;
	
	@Autowired
	private IDBasedFileAccessFactory idBasedFileAccessFactory;
	
	@Autowired(required=false)
	private CryptographicAwareIDBasedFileAccessFactory cryptographicAwareIDBasedFileAccessFactory;
	
	@Autowired(required=false)
	private CryptographicServiceAuthenticationFactory cryptographicServiceAuthenticationFactory;
	
	public DataSourceAccessDrive(ServerSession session) {
		super(session);
	}

	@Override
	public boolean canRead(String folderId, String id, String versionOrAttachmentId) throws DataSourceException {
		boolean canRead = false;

		final FileHelperService fileHelper = fileHelperServiceFactory.create(session);
        try {
            final int userId = sessionService.getUserId(session);
    		canRead = fileHelper.canReadFile(folderId, id, userId);
        } catch (Exception e) {
        	LOG.error("DataSourceAccessDrive exception caught trying to determine read access for user session", e);
        	throw new DataSourceException(ErrorCode.GENERAL_UNKNOWN_ERROR, e);
        }
        return canRead;
	}

	@Override
	public MetaData getMetaData(String folderId, String id, String versionOrAttachmentId, Optional<String> authCode) throws DataSourceException {
        final String encryptionInfo = determineEncryptionInfo(authCode);

		try {
            final String version = StringUtils.isEmpty(versionOrAttachmentId) ? FileStorageFileAccess.CURRENT_VERSION : versionOrAttachmentId;
			final IDBasedFileAccess fileAccess = getFileAccess(session, folderId, id, encryptionInfo);
	        final File fileMetaData = fileAccess.getFileMetadata(id, version);
	    	return mapFileMetaDataToMetaData(fileMetaData);
		} catch (OXException e) {
			final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, false);
			throw new DataSourceException(errorCode, e);
		}
	}

	@Override
	public InputStream getContentStream(String folderId, String id, String versionOrAttachmentId, Optional<String> authCode) throws DataSourceException {
        final String encryptionInfo = determineEncryptionInfo(authCode);

        try {
            final IDBasedFileAccess fileAccess = getFileAccess(session, folderId, id, encryptionInfo);
            final String version = (StringUtils.isEmpty(versionOrAttachmentId)) ? FileStorageFileAccess.CURRENT_VERSION : versionOrAttachmentId;
            final File metaData = fileAccess.getFileMetadata(id, version);
            final long fileSize = metaData.getFileSize();

            checkFilSizeAndMemoryConsumption(fileSize, encryptionInfo);

            InputStream inStream = null;
            try {
	        	inStream = fileAccess.getDocument(id, version);
	        	final byte[] data = IOUtils.toByteArray(inStream);
	        	return new ByteArrayInputStream(data);
	    	} catch (IOException e) {
	    		throw new DataSourceException(ErrorCode.LOADDOCUMENT_FAILED_ERROR, e);
	    	} finally {
	    		IOHelper.closeQuietly(inStream);
	    	}
		} catch (OXException e) {
			final ErrorCode errorCode = ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, false);
			throw new DataSourceException(errorCode, e);
		}
	}

    @Override
    public String getDataSource() {
        return DataSource.DRIVE;
    }

	private MetaData mapFileMetaDataToMetaData(File metaData) {
		return new MetaData(metaData.getFolderId(), metaData.getId(), metaData.getVersion(),
		                    metaData.getFileSize(), metaData.getFileName(), metaData.getFileMD5Sum(),
		                    metaData.getFileMIMEType());
	}

    private IDBasedFileAccess getFileAccess(ServerSession serverSession, String folderId, String fileId, String encryptionInfo) throws DataSourceException, OXException {
        if (null == idBasedFileAccessFactory)
            throw new DataSourceException(ErrorCode.GENERAL_SERVICE_DOWN_ERROR);

        if (null == serverSession)
            throw new DataSourceException(ErrorCode.GENERAL_SESSION_INVALID_ERROR);

        if (StringUtils.isEmpty(fileId) || StringUtils.isEmpty(folderId))
            throw new DataSourceException(ErrorCode.GENERAL_ARGUMENTS_ERROR);

        IDBasedFileAccess fileAccess = idBasedFileAccessFactory.createAccess(serverSession);
        if ((null != encryptionInfo) && (encryptionInfo.length() > 0)) {
            final CryptographicAwareIDBasedFileAccessFactory encryptionAwareFileAccessFactory = cryptographicAwareIDBasedFileAccessFactory;
            final CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = cryptographicServiceAuthenticationFactory;

            if ((encryptionAwareFileAccessFactory != null) && (encryptionAuthenticationFactory != null)) {
                final EnumSet<CryptographyMode> cryptMode = EnumSet.of(CryptographyMode.DECRYPT);
                final String authentication = encryptionInfo;
                fileAccess = encryptionAwareFileAccessFactory.createAccess(fileAccess, cryptMode, session, authentication );
            }
        }

        if (fileAccess == null)
            throw new DataSourceException(ErrorCode.GENERAL_SERVICE_DOWN_ERROR);

        return fileAccess;
    }

}
