/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.datasource.impl;

import java.util.Optional;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.openexchange.config.ConfigurationService;
import com.openexchange.office.datasource.access.DataSourceAccess;
import com.openexchange.office.datasource.access.DataSourceException;
import com.openexchange.office.datasource.access.DataSourceResourceAccess;
import com.openexchange.office.document.api.OXDocument;
import com.openexchange.office.tools.common.error.ErrorCode;
import com.openexchange.office.tools.common.memory.MemoryObserver;
import com.openexchange.office.tools.common.system.SystemInfoHelper;
import com.openexchange.office.tools.service.encryption.EncryptionInfo;
import com.openexchange.office.tools.service.encryption.EncryptionInfoService;
import com.openexchange.tools.session.ServerSession;

public abstract class DataSourceAccessBase implements DataSourceAccess {

    private static final int DEFAULT_MAX_DOCUMENT_FILE_SIZE = 100; // 100 MB

    @Autowired
    protected ConfigurationService configurationService;
    
    @Autowired
    protected EncryptionInfoService encryptionInfoService;
    
	protected ServerSession session;

	DataSourceAccessBase(ServerSession session) {
		this.session = session;
	}

	@Override
	public DataSourceResourceAccess createResourceAccess(String folderId, String id, String versionOrAttachmentId, Optional<String> authCode) {
		return new DataSourceResourceAccessImpl(this, folderId, id, versionOrAttachmentId, authCode);
	}

	@Override
	public int getContextId() {
		return session.getContextId();
	}

	protected String determineEncryptionInfo(Optional<String> authCode) {
		EncryptionInfo encryptionInfo = null;

		if ((authCode != null) && (authCode.isPresent())) {
            encryptionInfo = encryptionInfoService.createEncryptionInfo(session, authCode.get());
    		if (encryptionInfo.isPresent()) {
    			return encryptionInfo.getEncryptionValue();
    		}
		}
		return null;
	}

	protected boolean checkFilSizeAndMemoryConsumption(long fileSize, String encryptionInfo) throws DataSourceException {
        if (!isSupportedFileSize(fileSize))
            throw new DataSourceException(ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR);

        if (!enoughFreeHeapToLoadStreamData(fileSize, encryptionInfo))
        	throw new DataSourceException(ErrorCode.GENERAL_SYSTEM_BUSY_ERROR);

        return true;
	}

	protected boolean isSupportedFileSize(long fileSize) {
        long nMaxDocumentFileSize = DEFAULT_MAX_DOCUMENT_FILE_SIZE * 1024L * 1024L;

        if (null != configurationService) {
            nMaxDocumentFileSize = configurationService.getIntProperty("com.openexchange.office.maxDocumentFileSize", DEFAULT_MAX_DOCUMENT_FILE_SIZE) * 1024L * 1024L;
        }

        return (fileSize <= nMaxDocumentFileSize);
	}

    protected boolean enoughFreeHeapToLoadStreamData(long nFileSize, String encryptionInfo) {
        final MemoryObserver aMemObserver = MemoryObserver.get();

        final boolean bMemExceeded = aMemObserver.isUsageThresholdExceeded();
        final float fCurrentFileSizeFactor = StringUtils.isEmpty(encryptionInfo) ? 1.0f : OXDocument.PGP_FILE_SIZE_FACTOR;
        final long memNeededForStream = Math.round(nFileSize * fCurrentFileSizeFactor);
        final long freeMemAfterStreamLoaded = SystemInfoHelper.freeMemory() - memNeededForStream;
        return (!bMemExceeded && (freeMemAfterStreamLoaded > SystemInfoHelper.MINIMAL_FREE_BACKEND_HEAPSPACE));
    }

    protected String createHashFrom(long fileSize, String fileName, String mimeType) {
    	StringBuilder tmp = new StringBuilder(fileName);
    	tmp.append(fileSize);
    	tmp.append(mimeType);
    	return DigestUtils.md5Hex(tmp.toString());
    }
}
