/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imagemetadata.impl;

import java.awt.Dimension;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.drew.metadata.Directory;
import com.drew.metadata.Metadata;
import com.google.common.base.Throwables;
import com.openexchange.imageconverter.api.IMetadata;
import com.openexchange.imageconverter.api.IMetadataGroup;
import com.openexchange.imageconverter.api.MetadataGroup ;
import com.openexchange.imageconverter.api.MetadataKey;

/**
 * {@link MetadataImpl}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class MetadataImpl implements IMetadata {

    /**
     * Initializes a new {@link MetadataImpl}.
     */
    public MetadataImpl() {
        super();

        implInit();
    }

    /**
     * Initializes a new {@link MetadataImpl}.
     */
    public MetadataImpl(final Metadata metadata) {
        super();

        if (null != metadata) {
            for (final Directory curDirectory : metadata.getDirectories()) {
                final MetadataGroupImpl metadataGroupImpl = getOrCreateMetadataGroup(curDirectory);

                if (null != metadataGroupImpl) {
                    metadataGroupImpl.addDirectory(curDirectory);
                }
            }
        }

        implInit();
    }

    /**
     * Initializes a new {@link MetadataImpl}.
     */
    public MetadataImpl(final JSONObject jsonObject) {
        super();

        if (null != jsonObject) {
            for (final MetadataGroup  curDataGroup : MetadataGroup.values()) {
                JSONObject jsonMetadataGroupObj = jsonObject.optJSONObject(curDataGroup.getName());

                if (null != jsonMetadataGroupObj) {
                    getOrCreateMetadataGroup(curDataGroup).addJSONObject(jsonMetadataGroupObj);
                }
            }
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IMetadata#getImageDimension()
     */
    @Override
    public Dimension getImageDimension() {
        // The Header group is available at every instance
        final IMetadataGroup headerGroup = getMetadataGroup(MetadataGroup.HEADER);
        final Integer width = MetadataUtils.getIntegerValue(headerGroup, MetadataKey.HEADER_PIXEL_WIDTH);
        final Integer height = MetadataUtils.getIntegerValue(headerGroup, MetadataKey.HEADER_PIXEL_HEIGHT);

        return new Dimension(width.intValue(), height.intValue());
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IMetadata#getImageFormatShortName()
     */
    @Override
    public String getImageFormatName() {
        return getMetadataGroup(MetadataGroup.HEADER).getDataValue(MetadataKey.HEADER_FORMAT_NAME).toString();
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.api.IMetadata#getGroup(com.openexchange.imageconverter.api.api.MetadataGroup )
     */
    @Override
    public IMetadataGroup getMetadataGroup(MetadataGroup  metadataGroup) {
        return m_metadatGroupMap.get(metadataGroup);
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.api.IMetadata#iterator()
     */
    @Override
    public Collection<IMetadataGroup> getMetadataGroups() {
        return m_metadatGroupMap.values();
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.api.IMetadata#getJSONObject()
     */
    @Override
    public JSONObject getJSONObject() {
        final JSONObject ret = new JSONObject();

        try {
            for (final MetadataGroup  curGroup : m_metadatGroupMap.keySet()) {
                if (null != curGroup) {
                    ret.put(curGroup.getName(), m_metadatGroupMap.get(curGroup).getJSONObject());
                }
            }
        } catch (JSONException e) {
            LOG.error(Throwables.getRootCause(e).getMessage());
        }

        return ret;
    }

    // - Interface -------------------------------------------------------------

    /**
     * @param directory
     * @return
     */
    MetadataGroupImpl getOrCreateMetadataGroup(final Directory directory) {
        MetadataGroupImpl ret = null;

        if (MetadataUtils.isDirectorySupported(directory)) {
            ret = getOrCreateMetadataGroup(MetadataUtils.getDataGroupFromDirectory(directory));
        }

        return ret;
    }

    /**
     * @param directory
     * @return
     */
    MetadataGroupImpl getOrCreateMetadataGroup(final MetadataGroup metadataGroup) {
        MetadataGroupImpl ret = null;

        if (null != metadataGroup) {
            if (null == (ret = (MetadataGroupImpl) m_metadatGroupMap.get(metadataGroup))) {
                m_metadatGroupMap.put(metadataGroup, ret = new MetadataGroupImpl(metadataGroup));
            }
        }

        return ret;
    }

    // - Implementation --------------------------------------------------------

    private void implInit() {
        // at least the HEADER group ust exist in every case
        getOrCreateMetadataGroup(MetadataGroup.HEADER);
    }

    // - Members ---------------------------------------------------------------

    private final static Logger LOG = LoggerFactory.getLogger(MetadataImpl.class);

    private Map<MetadataGroup , IMetadataGroup> m_metadatGroupMap = new HashMap<>();
}