/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.json;

import java.io.File;
import java.io.InputStream;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.UUID;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.crypto.CryptographicServiceAuthenticationFactory;
import com.openexchange.documentconverter.DocumentConverterManager;
import com.openexchange.documentconverter.FileIdManager;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.MutableWrapper;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.Properties;
import com.openexchange.documentconverter.client.impl.ClientConfig;
import com.openexchange.documentconverter.client.impl.ClientManager;
import com.openexchange.file.storage.FileStorageCapability;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.FolderID;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographicAwareIDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographyMode;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.java.Streams;
import com.openexchange.mail.MailServletInterface;
import com.openexchange.mail.compose.Attachment;
import com.openexchange.mail.compose.CompositionSpaceService;
import com.openexchange.mail.compose.CompositionSpaces;
import com.openexchange.mail.dataobjects.MailPart;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.user.User;

/**
 * {@link DocumentConverterAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public abstract class DocumentConverterAction implements AJAXActionService {

    // - Statics ---------------------------------------------------------------

    // TODO (KA): check list of parameters for completeness prior to every release!!!
    // The parameter list is used for the fast detection of ManagedFiles, based on
    // the set of all given attributes to create a unique identifier. The ManagedFile itself
    // is created for the speedup of chunked requests, using the same source again and again
    final private static String[] idParamArray = { "uid", "id", "folder_id", "version", "revtag", "source", "attached", "module", "cryptoAuth", "attachmentId" };

    final protected static String FORMAT_JSON = "json";

    final protected static String FORMAT_FILE = "file";

    final protected static String JSON_KEY_ORIGIN = "origin";

    final protected static String JSON_KEY_ERRORCODE = "errorCode";

    final protected static String JSON_KEY_ERRORTEXT = "cause";

    final protected static String JSON_KEY_INVALID_CHARS = "invalidCharacters";

    final protected static String JSON_VALUE_DOUMENTCONVERTER = "DocumentConverter";

    /**
     * Initializes a new {@link DocumentConverterAction}.
     *
     * @param services
     */
    public DocumentConverterAction(final ServiceLookup services, final ClientManager manager, final FileIdManager fileIdManager, final ClientConfig clientConfig) {
        super();

        m_services = services;
        m_manager = manager;
        m_fileIdManager = fileIdManager;
        m_clientConfig = clientConfig;
    }

    /**
     * @param request
     * @param session
     * @param jobError
     * @return
     */
    protected File getRequestSourceFile(
        @NonNull final AJAXRequestData request,
        @NonNull final ServerSession session,
        @NonNull final MutableWrapper<Boolean> documentDecrypted,
        @NonNull final MutableWrapper<JobError> jobError) {

        final String documentId = getDocumentId(request);
        File ret = null;

        try {
            m_fileIdManager.lock(documentId);

            if (null == (ret = m_fileIdManager.getDataFile(documentId))) {
                final String fileId = request.getParameter("id");
                final String attachmentId = request.getParameter("attachmentId");
                final String source = request.getParameter("source");
                final String attached = request.getParameter("attached");
                final String module = request.getParameter("module");
                final String folderId = request.getParameter("folder_id");
                final String version = request.getParameter("version");
                final String cryptoAuth = request.getParameter("cryptoAuth");
                final boolean mailCheck = ((attached != null) && (folderId != null) && (attached.length() > 0));
                final boolean taskCheck = mailCheck && (module != null);
                final boolean composeCheck = (fileId != null) && (attachmentId != null);
                MailServletInterface mailServletInterface = null; // closing done in finally block
                InputStream documentStm = null; // closing done in finally block

                try {
                    if (mailCheck && "mail".equals(source)) {
                        // mail attachment

                        if (StringUtils.isNotEmpty(cryptoAuth)) {
                            // If GuardMail, pull authentication from request
                            final CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = m_services.getOptionalService(CryptographicServiceAuthenticationFactory.class);
                            String authentication = null;

                            if (null != encryptionAuthenticationFactory) {
                                authentication = encryptionAuthenticationFactory.createAuthenticationFrom(request.optHttpServletRequest());
                            }

                            mailServletInterface = MailServletInterface.getInstanceWithDecryptionSupport(session, authentication);
                            documentDecrypted.set(Boolean.TRUE);
                        } else {
                            mailServletInterface = MailServletInterface.getInstance(session);
                        }

                        if (null != mailServletInterface) {
                            final MailPart mailPart = mailServletInterface.getMessageAttachment(folderId, fileId, attached, false);

                            if (null != mailPart) {
                                documentStm = mailPart.getInputStream();
                            }
                        }
                    } else if (composeCheck && "compose".equals(source)) {
                        // Composition space attachment
                        CompositionSpaceService compositionSpaceService = m_services.getOptionalService(CompositionSpaceService.class);
                        if (compositionSpaceService == null) {
                            jobError.set(JobError.GENERAL);
                        } else {
                            UUID compositionSpaceUuid = CompositionSpaces.parseCompositionSpaceId(fileId);
                            UUID attachmentUuid = CompositionSpaces.parseAttachmentId(attachmentId);
                            Attachment attachment = compositionSpaceService.getAttachment(compositionSpaceUuid, attachmentUuid, session);
                            documentStm = attachment.getData();
                        }
                    } else if (taskCheck && ("tasks".equals(source) || "calendar".equals(source) || "contacts".equals(source))) {
                        // other attachment from tasks, calendar or contact

                        documentStm = Attachments.getInstance().getAttachedFile(
                            session,
                            Integer.parseInt(folderId),
                            Integer.parseInt(attached),
                            Integer.parseInt(module),
                            Integer.parseInt(fileId),
                            session.getContext(),
                            session.getUser(),
                            session.getUserConfiguration());
                    } else {
                        // use FileStore as default data source
                        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                        IDBasedFileAccess fileAccess = (null != fileFactory) ? fileFactory.createAccess(session) : null;

                        if ((null != fileAccess) && "guard".equals(source)) {
                            if (StringUtils.isNotEmpty(cryptoAuth)) {
                                // If GuardMail, get authentication
                                final CryptographicAwareIDBasedFileAccessFactory encryptionAwareFileAccessFactory = m_services.getOptionalService(CryptographicAwareIDBasedFileAccessFactory.class);
                                final CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = m_services.getOptionalService(CryptographicServiceAuthenticationFactory.class);

                                if ((null != encryptionAwareFileAccessFactory) && (null != encryptionAuthenticationFactory)) {
                                    final EnumSet<CryptographyMode> cryptMode = EnumSet.of(CryptographyMode.DECRYPT);

                                    fileAccess = encryptionAwareFileAccessFactory.createAccess(fileAccess, cryptMode, session,
                                        encryptionAuthenticationFactory.createAuthenticationFrom(request.optHttpServletRequest()));

                                    documentDecrypted.set(Boolean.TRUE);
                                } else {
                                    // file is encrypted, but we can't decrypt...
                                    fileAccess = null;
                                }
                            } else {
                                fileAccess = null;
                            }

                            // set output paramater jobError to JobError.PASSWORD in
                            // case we got no valid fileAccess for given Guard parameters
                            if (null == fileAccess) {
                                jobError.set(JobError.PASSWORD);
                            }
                        }

                        if (null != fileAccess) {
                            // #46555 map version String("null") to null
                            String versionToUse = "null".equalsIgnoreCase(version) ? FileStorageFileAccess.CURRENT_VERSION : version;

                            // #46555 check for versioning support
                            if (FileStorageFileAccess.CURRENT_VERSION != versionToUse) {
                                final FolderID folderIdentifier = new FolderID(folderId);

                                if (!fileAccess.supports(folderIdentifier.getService(), folderIdentifier.getAccountId(), FileStorageCapability.FILE_VERSIONS)) {
                                    versionToUse = FileStorageFileAccess.CURRENT_VERSION;
                                }
                            }

                            documentStm = fileAccess.getDocument(fileId, versionToUse);
                        }
                    }

                    if (null != documentStm) {
                        ret = m_fileIdManager.setDataInputStream(documentId, documentStm);
                    }
                } catch (final Exception e) {
                    DocumentConverterManager.logExcp(e);
                } finally {
                    Streams.close(documentStm);

                    if (mailServletInterface != null) {
                        mailServletInterface.close();
                    }
                }
            }
        } finally {
            m_fileIdManager.unlock(documentId);
        }

        // set output parameter for decrypted document to
        // FALSE if we have no document to return at all
        if (null == ret) {
            documentDecrypted.set(Boolean.FALSE);
        }

        return ret;
    }

    /**
     * @param request
     * @param session
     * @param isAsync
     * @param resultJobError
     * @return
     */
    protected File getPDFFile(
        @NonNull final AJAXRequestData request,
        @NonNull final ServerSession session,
        final boolean isAsync,
        @NonNull final MutableWrapper<Boolean> documentDecrypted,
        @NonNull final MutableWrapper<String> resultJobId,
        @NonNull final MutableWrapper<JobError> resultJobError) {

        final IDocumentConverter docConverter = m_services.getService(IDocumentConverter.class);
        File resultFile = null;

        resultJobError.set(JobError.GENERAL);

        if (null != docConverter) {
            final String locale = getPreferredLanguage(session);

            if (null == resultJobId.get()) {
                resultJobId.set(getDocumentId(request, "pdf", locale));
            }

            try {
                m_fileIdManager.lock(resultJobId.get());

                if (null == (resultFile = m_fileIdManager.getDataFile(resultJobId.get()))) {
                    final File sourceFile = getRequestSourceFile(request, session, documentDecrypted, resultJobError);

                    if(null != sourceFile) {
                        final String fileName = request.getParameter("filename");

                        if ((null != fileName) && "pdf".equalsIgnoreCase(FilenameUtils.getExtension(fileName))) {
                            // return source file in case of a PDF source
                            try (final InputStream srcInputStm = FileUtils.openInputStream(sourceFile)) {
                                resultFile = m_fileIdManager.setDataInputStream(resultJobId.get(), srcInputStm);
                                resultJobError.set(JobError.NONE);
                            }
                        } else {
                            // convert to PDF via DocumentConverter
                            if (isAsync && documentDecrypted.get().booleanValue()) {
                                // we don't trigger any async conversion in case
                                // of a decrypted document and leave ASAP since
                                // we don't want to cache decrypted document
                                // conversion results for security reasons
                                resultJobError.set(JobError.NONE);
                            } else {
                                final HashMap<String, Object> jobProperties = new HashMap<>(12);
                                final HashMap<String, Object> resultProperties = new HashMap<>(8);

                                jobProperties.put(Properties.PROP_INPUT_FILE, sourceFile);
                                jobProperties.put(Properties.PROP_PRIORITY, JobPriority.fromString(request.getParameter("priority")));
                                jobProperties.put(Properties.PROP_LOCALE, locale);

                                if (null != fileName) {
                                    jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
                                }

                                if (isAsync) {
                                    jobProperties.put(Properties.PROP_ASYNC, Boolean.TRUE);
                                    jobProperties.put(Properties.PROP_PRIORITY, JobPriority.BACKGROUND);
                                } else {
                                    // this is a user request in synchronous case
                                    jobProperties.put(Properties.PROP_USER_REQUEST, Boolean.TRUE);
                                }

                                if (documentDecrypted.get().booleanValue()) {
                                    // set 'NO_CACHE=true' property for decrypted documents since we don't
                                    // want to cache decrypted document conversion results for security reasons
                                    jobProperties.put(Properties.PROP_NO_CACHE, Boolean.TRUE);
                                }

                                try (InputStream pdfResultStm = docConverter.convert("pdf", jobProperties, resultProperties)) {
                                    if (null != pdfResultStm) {
                                        resultFile = m_fileIdManager.setDataInputStream(resultJobId.get(), pdfResultStm);
                                        resultJobError.set(JobError.NONE);
                                    } else {
                                        JobError pdfJobError = JobError.fromObject(resultProperties.get(Properties.PROP_RESULT_ERROR_CODE));

                                        if (JobError.NONE == pdfJobError) {
                                            pdfJobError = JobError.GENERAL;
                                        }

                                        resultJobError.set(isAsync ? JobError.NONE : pdfJobError);
                                    }
                                }
                            }
                        }
                    } else {
                        // async case for PDF documents
                        resultJobError.set(isAsync ? JobError.NONE : JobError.NO_CONTENT);
                    }
                } else {
                    resultJobError.set(JobError.NONE);
                }
            } catch (Exception e) {
                DocumentConverterManager.logExcp(e);
            } finally {
                m_fileIdManager.unlock(resultJobId.get());
            }
        }

        return resultFile;
    }

    /**
     * @param request
     * @param fileHolder
     * @return
     */
    AJAXRequestResult getRequestResult(@NonNull AJAXRequestData request, IFileHolder fileHolder, JSONObject jsonObject, JobError jobError) {
        AJAXRequestResult ret = null;

        if ((jobError == JobError.NONE) && (null != fileHolder)) {
            request.setFormat(FORMAT_FILE);
            ret = new AJAXRequestResult(fileHolder, FORMAT_FILE);
        } else {
            final JSONObject jsonResult = (null != jsonObject) ? jsonObject : new JSONObject();

            try {
                jsonResult.put(JSON_KEY_ORIGIN, JSON_VALUE_DOUMENTCONVERTER);

                if ((null != jobError) && (jobError != JobError.NONE) && !jsonResult.has(JSON_KEY_ERRORTEXT)) {
                    jsonResult.put(JSON_KEY_ERRORTEXT,  jobError.errorText());
                }
            } catch (JSONException e) {
                DocumentConverterManager.logExcp(e);
            }

            request.setFormat(FORMAT_JSON);
            ret = new AJAXRequestResult(jsonResult, FORMAT_JSON);
        }

        return ret;
    }

    /**
     * @param session
     * @return
     */
    protected static @NonNull String getPreferredLanguage(ServerSession session) {
        final User sessionUser = session.getUser();
        String prefLanguage = null;

        // user language
        if (null != sessionUser) {
            prefLanguage = sessionUser.getPreferredLanguage();
        }

        return StringUtils.isBlank(prefLanguage) ? "en_US" : prefLanguage;
    }

    /**
     * @param request
     * @return
     */
    protected static @NonNull String getDocumentId(@NonNull AJAXRequestData request, String... additionalData) {
        final StringBuilder documentIdBuilder = new StringBuilder(256);

        for (final String[] curParamArray : new String[][] { idParamArray, additionalData }) {
            if (null != curParamArray) {
                final boolean isParam = curParamArray.equals(idParamArray);

                for (final String curValue : curParamArray) {
                    documentIdBuilder.append(isParam ? request.getParameter(curValue) : curValue).append('~');
                }
            }
        }

        return documentIdBuilder.toString();
    }

    // - Members ---------------------------------------------------------------

    final protected ServiceLookup m_services;

    final protected ClientManager m_manager;

    final protected ClientConfig m_clientConfig;

    final protected FileIdManager m_fileIdManager;
}
