/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.user;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.Validate;

/**
 * Manages the user data for every member of a connection instance.
 *
 * @author Carsten Driesner
 * @since 7.8.0
 */
public class UserDataManager {

    private final Map<String, UserData> userDataMap = new HashMap<>();
    private UserData lastUserData = null;
    private String lastUserKey = null;

    /**
     * Initalizes a new UserDataManager instance.
     */
    public UserDataManager() {
        // nothing to do
    }

    public synchronized int getCount() {
        return userDataMap.size();
    }

    public synchronized int getCountOnNode(String sNodeUID) {
    	Validate.notNull(sNodeUID);

    	int result = 0;

        for (final UserData aUserData : userDataMap.values()) {
            result += (sNodeUID.equals(aUserData.getNodeUUID())) ? 1 : 0;
        }

        return result;
    }

    /**
     * Provides a shallow copy of all users/clients known by this
     * instance.
     *
     * @return
     */
    public synchronized List<UserData> getUserData() {
        return new ArrayList<>(userDataMap.values());
    }

    public synchronized List<UserData> getUserDataCloned() {
		final List<UserData> result = new ArrayList<>();

		final Collection<UserData> orgUserDataList = userDataMap.values();
		for (final UserData orgUserData : orgUserDataList) {
			result.add(orgUserData.getCopy());
		}
		return result;
    }

    public synchronized void updateUserData(Collection<String> userIds) {
    	userDataMap.keySet().retainAll(userIds);
    }
    
    /**
     * Returns the user uuids which fulfill the provided
     * @param aUserDataFilter
     * @return
     */
    public synchronized List<String> getFilteredUserData(Predicate<UserData> aUserDataFilter) {
        return userDataMap.values().stream().filter(aUserDataFilter)
                                            .map(o -> o.getUserId())
                                            .collect(Collectors.toList());
    }

    /**
     * Determines if user-specific data is associated with the real-time id.
     *
     * @param id the real-time id of the user.
     * @return true if user data is available and false if not
     */
    public synchronized boolean hasUserData(final String id) {
        Validate.notNull(id);

        return userDataMap.containsKey(id) || id.equalsIgnoreCase(lastUserKey);
    }

    /**
     * Retrieves the user-specific data associated with the real-time id.
     *
     * @param id the real-time id of the user.
     * @return the user-specific data or null
     */
    public final synchronized UserData getUserData(final String id) {
        Validate.notNull(id);

        if (!userDataMap.isEmpty()) {
            return userDataMap.get(id);
        } else if (id.equalsIgnoreCase(lastUserKey)) {
            return lastUserData;
        }

        return null;
    }

    /**
     * Retrieves the user-specific data associated with the real-time id if
     * that user is still a member of this document group.
     *
     * @param id the real-time id of the user.
     * @return the user-specific data or null
     */
    public final synchronized UserData getMemberUserData(final String id) {
        Validate.notNull(id);

        if (!userDataMap.isEmpty())
            return userDataMap.get(id);

        return null;
    }

    /**
     * Stores the user-specific data associated with the real-time id.
     *
     * @param id the real-time id of the user.
     * @param userData the user-specific data to be stored in association with the real-time id
     */
    public synchronized void addUser(final String id, final UserData userData) {
        Validate.notNull(id);

        userData.setUserId(id);
        userDataMap.put(id, userData);
    }

    /**
     * Removes the user-specific data associated with a real-time id.
     *
     * @param id the real-time id where the user-specific data should be removed.
     */
    public synchronized void removeUser(final String id) {
        Validate.notNull(id);

        final UserData userData = userDataMap.remove(id);
        if (userDataMap.isEmpty()) {
            // store last user data for use after the client left, e.g. dispose
            lastUserData = userData;
            lastUserKey = id;
        }
    }

}
