/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.error;

import com.openexchange.exception.Category;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageExceptionCodes;
import com.openexchange.filestore.FileStorageCodes;
import com.openexchange.filestore.QuotaFileStorageExceptionCodes;
import com.openexchange.folderstorage.FolderExceptionErrorMessage;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;

/**
 * A helper class to map certain exception codes to specific error codes
 * defined in the enumation ErrorCode.
 *
 * {@link ErrorCode}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */

public class ExceptionToErrorCode {

    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(ExceptionToErrorCode.class);

    /**
     * Maps certain OXExceptions to specific error codes.
     *
     * @param e
     *  An instance of a OXException.
     *
     * @param defErrorCode
     *  The default error code to be used if there is no mapping available.
     *
     * @param logEx
     *  Specifies if the exception should be logged or not.
     *
     * @return
     *  The specific error code or the default if no mapping is available.
     */
    static public ErrorCode map(OXException e, ErrorCode defErrorCode, boolean logEx) {
        ErrorCode errorCode = defErrorCode;
        String msg = "Exception caught";
        final String prefix = e.getPrefix();
        final int code = e.getCode();
        final Category category = e.getCategory();

        switch (prefix) {
            case "DBP":
            	// database exceptions
            	if (category == Category.CATEGORY_SERVICE_DOWN) {
            		errorCode = ErrorCode.GENERAL_SERVICE_DOWN_ERROR;
            		msg = "RT Connection: Database service not cannot be contacted!";
            	} else {
            		errorCode = ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR;
            		msg = "RT Connection: Database error detected!";
            	}
            	break;
            case "FLS":
                // special handling for file storage exceptions
                if (code == QuotaFileStorageExceptionCodes.STORE_FULL.getNumber()) {
                    // set specific error code if quota reached
                    errorCode = ErrorCode.GENERAL_QUOTA_REACHED_ERROR;
                    msg = "RT connection: Quota reached error detected!";
                } else if (code == FileStorageCodes.FILE_NOT_FOUND.getNumber()) {
                    errorCode = ErrorCode.GENERAL_FILE_NOT_FOUND_ERROR;
                    msg = "RT connection: File not found error detected!";
                }
                break;
            case "FLD":
                if (code == FolderExceptionErrorMessage.FOLDER_NOT_VISIBLE.getNumber()) {
                    errorCode = ErrorCode.GENERAL_FOLDER_NOT_VISIBLE_ERROR;
                    msg = "RT connection: Folder not visible, no permissions!";
                } else if (code == FolderExceptionErrorMessage.NOT_FOUND.getNumber()) {
                    errorCode = ErrorCode.GENERAL_FOLDER_NOT_FOUND_ERROR;
                    msg = "RT connection: Folder not found, removed folder?";
                } else if (code == FolderExceptionErrorMessage.INVALID_FOLDER_ID.getNumber()) {
                    errorCode = ErrorCode.GENERAL_FOLDER_INVALID_ID_ERROR;
                    msg = "RT connection: Folder id is invalid!";
                } else {
                    errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                    msg = "RT connection: Unkown folder error occurred!";
                }
                break;
            case "QUOTA":
                // set specific error code if quota reached
                errorCode = ErrorCode.GENERAL_QUOTA_REACHED_ERROR;
                msg = "RT connection: Quota reached error detected!";
                break;
            case "IFO":
                if (code == InfostoreExceptionCodes.NO_READ_PERMISSION.getNumber()) {
                    // read permissions missing
                    errorCode = ErrorCode.GENERAL_PERMISSION_READ_MISSING_ERROR;
                    msg = "RT connection: No read permissions detected!";
                } else if (code == InfostoreExceptionCodes.NO_WRITE_PERMISSION.getNumber()) {
                    // read permissions missing
                    errorCode = ErrorCode.GENERAL_PERMISSION_WRITE_MISSING_ERROR;
                    msg = "RT connection: No write permissions!";
                } else if (code == InfostoreExceptionCodes.NOT_EXIST.getNumber()) {
                    // file not found
                    errorCode = ErrorCode.GENERAL_FILE_NOT_FOUND_ERROR;
                    msg = "RT connection: Original document couldn't be found!";
                } else if (code == InfostoreExceptionCodes.DOCUMENT_NOT_EXIST.getNumber()) {
                    // file not found
                    errorCode = ErrorCode.GENERAL_FILE_NOT_FOUND_ERROR;
                    msg = "RT connection: Original document couldn't be found!";
                } else if (code == InfostoreExceptionCodes.ALREADY_LOCKED.getNumber()) {
                    // file is locked now
                    errorCode = ErrorCode.GENERAL_FILE_LOCKED_ERROR;
                    msg = "RT connection: The document is locked and cannot be modified";
                } else if (code == InfostoreExceptionCodes.VALIDATION_FAILED_CHARACTERS.getNumber()) {
                    // illegal character in file name
                    errorCode = ErrorCode.RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR;
                    msg = "RT connection: Renamed failed due to invalid character in file name";
                    // try to retrieve the illegal character from the exception
                    errorCode.setValue(getFirstArg(e));
                }
                break;
            case "FILE_STORAGE":
                if (code == FileStorageExceptionCodes.FILE_NOT_FOUND.getNumber()) {
                    // file not found error
                    errorCode = ErrorCode.GENERAL_FILE_NOT_FOUND_ERROR;
                    msg = "RT connection: File not found error detected!";
                } else if (code == FileStorageExceptionCodes.ILLEGAL_CHARACTERS.getNumber()) {
                    errorCode = ErrorCode.RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR;
                    msg = "RT connection: Rename not possible due to illegal characters!";
                    // try to retrieve the illegal character from the exception
                    errorCode.setValue(getFirstArg(e));
                }
                break;
            case "DROPBOX":
                // special code to detect specific dropbox errors
                if (code == 2) {
                    errorCode = ErrorCode.LOADDOCUMENT_TIMEOUT_RETRIEVING_STREAM_ERROR;
                    msg = "RT connection: Couldn't retrieve stream from Dropbox in time";
                }
                break;
            default:
                // nothing to do default values set
                break;
        }

        if (logEx) {
            LOG.error(msg, e);
        }

        return errorCode;
    }

    /**
     * Retrieve the first display argument from the exception.
     *
     * @param e the exception
     * @return
     */
    static private String getFirstArg(final OXException e) {
    	String result = null;

    	final Object[] dispArgs = e.getDisplayArgs();

    	if (dispArgs.length > 0) {
    		if (dispArgs[0] instanceof String)
    			result = (String)dispArgs[0];
    		else if (dispArgs[0] != null) {
    			result = dispArgs[0].toString();
    		}
    	}

    	return result;
    }
}
