/*
 *
 *    OPEN-XCHANGE legal information *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.database;

import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.tools.annotation.NonNull;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.pool.HikariPool;

import liquibase.Liquibase;
import liquibase.changelog.ChangeSet;
import liquibase.changelog.ChangeSet.ExecType;
import liquibase.changelog.ChangeSet.RunStatus;
import liquibase.changelog.DatabaseChangeLog;
import liquibase.changelog.visitor.ChangeExecListener;
import liquibase.database.Database;
import liquibase.database.jvm.JdbcConnection;
import liquibase.exception.LiquibaseException;

/**
 * {@link DocumentsDatabase}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class DocumentsDatabase implements ChangeExecListener {

    /**
     * Initializes a new {@link DocumentsDatabase}.
     */
    public DocumentsDatabase(
        final DatabaseResourceAccessor resourceAccessor,
        final ConnectionData readConnectionData,
        final ConnectionData writeConnectionData,
        boolean updateSchema) throws DatabaseException {

        this.resourceAccessor = resourceAccessor;

        this.readConnectionData = readConnectionData;
        this.writeConnectionData = (null != writeConnectionData) ? writeConnectionData : readConnectionData;


        if (isEmpty(this.readConnectionData.getConnectionURL()) ) {

            throw new DatabaseException("Property readUrl must not be empty (Example Url: jdbc:mysql://host:port/databasename)");
        }

        // create connection pools
        implCreateConnectionPools(readConnectionData != null, writeConnectionData != null);

        // perform Liquibase update
        if (updateSchema) {
        	implUpdateDatabase();
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#willRun(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database, liquibase.changelog.ChangeSet.RunStatus)
     */
    @Override
    public void willRun(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database, RunStatus runStatus) {
        if (null != changeSet) {
            implLogChangeEvent(runStatus, changeSet.getDescription());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#ran(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database, liquibase.changelog.ChangeSet.ExecType)
     */
    @Override
    public void ran(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database, ExecType execType) {
        if (null != changeSet) {
            implLogChangeEvent(execType, changeSet.getDescription());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see liquibase.changelog.visitor.ChangeExecListener#rolledBack(liquibase.changelog.ChangeSet, liquibase.changelog.DatabaseChangeLog, liquibase.database.Database)
     */
    @Override
    public void rolledBack(ChangeSet changeSet, DatabaseChangeLog databaseChangeLog, Database database) {
        if (null != changeSet) {
            implLogChangeEvent(null, changeSet.getDescription());
        }
    }

    // - Database access methods -----------------------------------------------

    /**
     * getReadConnection
     */
    public Connection getReadConnection() throws DatabaseException {
        if (null == this.readConnectionPool) {
            throw new DatabaseException("No read connection pool available. Please check logging output and setup!");
        }

        try {
            return this.readConnectionPool.getConnection();
        } catch (SQLException e) {
            throw new DatabaseException("Exception caught when getting read connection from connectiin pool", e);
        }
    }

    //  - Implementation -------------------------------------------------------

    /**
     * getWriteConnection
     */
    public Connection getWriteConnection() throws DatabaseException {
        if (null == this.writeConnectionPool) {
            throw new DatabaseException("No write connection pool available. Please check logging output and setup!");
        }

        try {
            return this.writeConnectionPool.getConnection();
        } catch (SQLException e) {
            throw new DatabaseException("Exception caught when getting write connection from connection pool", e);
        }
    }

    /**
     * @throws DatabaseException
     * @throws LiquibaseException
     */
    private void implUpdateDatabase() throws DatabaseException {
        final String resourceName = this.resourceAccessor.getResourceName();

        try (final Connection con = getWriteConnection()) {
            final Liquibase liquibase = new Liquibase(resourceName, this.resourceAccessor, new JdbcConnection(con));

            liquibase.setChangeExecListener(this);
            liquibase.update(null);
        } catch (Exception e) {
            throw new DatabaseException("Exception caught when updating database via Liquibase resource: " + resourceName, e);
        }
    }

    /**
     * @throws DatabaseException
     */
    private void implCreateConnectionPools(boolean createReadConn, boolean createWriteConn) throws DatabaseException {
        this.readConnectionPool = implCreateConnectionPool(this.readConnectionData, this.readConnectionData, true);
        this.writeConnectionPool = implCreateConnectionPool(this.writeConnectionData, this.writeConnectionData, false);
    }

    /**
     * @param connectionData
     * @param poolData
     * @param readOnly
     * @return
     * @throws DatabaseException
     */
    private static HikariPool implCreateConnectionPool(@NonNull final DatabaseConnectionData connectionData,
        @NonNull final ConnectionPoolData poolData, boolean readOnly) throws DatabaseException {

        final HikariConfig poolConfig = new HikariConfig();
        final Properties databaseProperties = connectionData.getProperties();
        final String driverClassName = connectionData.getDriverClassName();

        try {
            Class.forName(driverClassName);
        } catch (ClassNotFoundException e) {
            throw new DatabaseException("Exception caught when getting JDBC driver runtime class: " + driverClassName, e);
        }

        // DB related Hikari properties
        poolConfig.setJdbcUrl(connectionData.getConnectionURL());
        poolConfig.setUsername(connectionData.getUserName());
        poolConfig.setPassword(connectionData.getPassword());
        poolConfig.setAutoCommit(false);
        poolConfig.setReadOnly(readOnly);

        // common DB properties
        for (final String curPropKey : databaseProperties.stringPropertyNames()) {
            poolConfig.addDataSourceProperty(curPropKey, databaseProperties.getProperty(curPropKey));
        }

        // Hikari pool properties
        poolConfig.setMaximumPoolSize(poolData.getConnectionPoolSize());
        poolConfig.setConnectionTimeout(poolData.getConnectionPoolConnectTimeoutMillis());
        poolConfig.setIdleTimeout(poolData.getConnectionPoolIdleTimeoutMillis());

        HikariPool result = null;
        try {
            result = new HikariPool(poolConfig);
        } catch (final Exception e) {
            final Map<String, String> props = new HashMap<>();
            props.put("driverClassName", poolConfig.getDriverClassName());
            props.put("jdbcUrl", poolConfig.getJdbcUrl());
            props.put("username", poolConfig.getUsername());
            props.put("password", StringUtils.isNotEmpty(poolConfig.getPassword()) ? "<set>" : "<empty>");
            throw new DatabaseException("Exception caught trying to create database connection pool", e, props);
        }

        return result;
    }

    /**
     * @param status
     * @param description
     */
    private static void implLogChangeEvent(final Object status, final String description) {
        if (LOG.isInfoEnabled()) {
            if (null != status) {
                LOG.debug("FileItemService database update received status: " + status.toString());
            }

            if (isNotEmpty(description)) {
                LOG.debug("FileItemService database update description: " + description);
            }
        }
    }

    // - Static members --------------------------------------------------------

    final private static Logger LOG = LoggerFactory.getLogger(DocumentsDatabase.class);

    // - Members ---------------------------------------------------------------

    private DatabaseResourceAccessor resourceAccessor;

    private ConnectionData readConnectionData;

    private ConnectionData writeConnectionData;

    private HikariPool readConnectionPool = null;

    private HikariPool writeConnectionPool = null;
}
