/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.database;

import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.util.Map;
import java.util.Properties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Throwables;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.WildcardNamePropertyFilter;
import com.openexchange.exception.OXException;
import com.openexchange.office.tools.annotation.NonNull;

/**
 * {@link DatabaseConfiguration}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.1
 */
public class DatabaseConfiguration implements ConnectionData {

    /**
     * Initializes a new {@link DatabaseConfiguration}.
     *
     * The connection data properties are read from the given {@link ConfigurationService} interface.
     * All dabatbase related connection properties are prefixed with the given {@link String} parameter.
     *
     * The properties that are read from the configuration are listed below:
     *  - ${configPrefix}.driverClass (mandatory)
     *  - ${configPrefix}.connectionURL (mandatory)
     *  - ${configPrefix}.userName (mandatory)
     *  - ${configPrefix}.password (mandatory)
     *  - ${configPrefix}.property.[0-9]+ (optional)
     *  - ${configPrefix}.connectionPool.maxPoolSize (optional)
     *  - ${configPrefix}.connectionPool.connectTimeoutMillis (optional)
     *  - ${configPrefix}.connectionPool.idleTimeoutMillis (optional)
     */
    public DatabaseConfiguration(@NonNull final ConfigurationService configService, @NonNull final String configPrefix) {
        super();

        final StringBuilder configPrefixBuilder = new StringBuilder(configPrefix.length() + 16);

        configPrefixBuilder.append(configPrefix);

        if (!configPrefix.endsWith(".")) {
            configPrefixBuilder.append('.');
        }

        this.configService = configService;
        this.configPrefix = configPrefixBuilder.toString();
    }

    /**
     * @return The driver class for the database connection to be used.
     */
    @Override
    public String getDriverClassName() {
        return this.configService.getProperty(this.configPrefix + "driverClass",
            ConnectionData.super.getDriverClassName());
    }

    /**
     * @return The URL for the database connection to be used.
     */
    @Override
    public String getConnectionURL() {
        return this.configService.getProperty(this.configPrefix + "connectionURL");
    }

    /**
     * @return The database username for the connection.
     */
    @Override
    public String getUserName() {
        return this.configService.getProperty(this.configPrefix + "userName");
    }

    /**
     * @return The database password for the connection.
     */
    @Override
    public String getPassword() {
        return this.configService.getProperty(this.configPrefix + "password");
    }

    /**
     * @return Database specific {@link Properties}, containing
     *  {@link String} keys and {@link String}/{@link Boolean}/{@link Integer}
     *  object values throughout.
     */
    @Override
    public Properties getProperties() {
        Properties properties = null;

        try {
            final Map<String, String> propertiesMap = this.configService.getProperties(
                new WildcardNamePropertyFilter(this.configPrefix + "property.*"));

            properties = new Properties();

            for (final String curKey : propertiesMap.keySet()) {
                final String configPair = propertiesMap.get(curKey);

                if (isNotEmpty(configPair)) {
                    final int assignPos = configPair.lastIndexOf('=');

                    if ((assignPos > 0) && (assignPos < (configPair.length() - 1))) {
                        properties.put(configPair.substring(0, assignPos), configPair.substring(assignPos + 1));
                    }
                }
            }
        } catch (OXException e) {
            LOG.error("Exception caught when reading Documents database properties: " + Throwables.getRootCause(e) + " using defaults!");
            properties = ConnectionData.super.getProperties();
        }

        return properties;
    }

    /**
     * @return The number of open connections within the database connection pool.
     */
    @Override
    public int getConnectionPoolSize() {
        return this.configService.getIntProperty(this.configPrefix + "connectionPool.maxPoolSize",
            ConnectionData.super.getConnectionPoolSize());
    }

    /**
     * @return The connection timeout of a pooled connection.
     *  Given in milliseconds.
     */
    @Override
    public long getConnectionPoolConnectTimeoutMillis() {
        return this.configService.getIntProperty(this.configPrefix + "connectionPool.connectTimeoutMillis",
            (int) ConnectionData.super.getConnectionPoolConnectTimeoutMillis());
    }

    /**
     * @return The idle timeout of a pooled connection, after which an idling
     *  connection is closed and removed from the pool. Given in milliseconds.
     */
    @Override
    public long getConnectionPoolIdleTimeoutMillis() {
        return this.configService.getIntProperty(this.configPrefix + "connectionPool.idleTimeoutMillis",
            (int) ConnectionData.super.getConnectionPoolIdleTimeoutMillis());
    }

    // - Public API ------------------------------------------------------------

    /**
     * @return
     */
    public ConfigurationService getConfigurationService() {
        return this.configService;
    }

    /**
     * @return
     */
    public String getConfigPrefix() {
        return this.configPrefix;
    }

    // - Members ---------------------------------------------------------------

    private static Logger LOG = LoggerFactory.getLogger(DatabaseConfiguration.class);

    private ConfigurationService configService;

    private String configPrefix;

}
