/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.spellchecker.hunspell;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.spellchecker.hunspell.jna.Hunspell;

/**
 * {@link DictionaryContainer}
 *
 * @author <a href="mailto:oliver.specht@open-xchange.com">Firstname Lastname</a>
 */
public class DictionaryContainer {
    final private static Logger LOG = LoggerFactory.getLogger(DictionaryContainer.class);

    private final List<Hunspell.Dictionary> dictionaries = new ArrayList<Hunspell.Dictionary>();
    private final List<String> locales = new ArrayList<String>();
    /**
     * Initializes a new {@link DictionaryContainer}.
     * @param hunspell instance of Hunspell
     * @param dictPath path to the Hunspell dictionaries
     */
    public DictionaryContainer( Hunspell hunspell, String dictPath ){
        FileFilter fileFilter = new FileFilter() {
            @Override
            public boolean accept(File file) {
                if( file.isDirectory() )
                    return false;
                String fileName = file.toURI().getPath();
                return fileName.endsWith(".dic");
            }
        };
        File dictDirectory = new File( dictPath );
        // get all .dic and .aff files from the folder
        File files[] = dictDirectory.listFiles( fileFilter );
        for( int dictIndex = 0; dictIndex < files.length; ++ dictIndex ){
            File dictFile = files[dictIndex];
            String fileName = dictFile.getName();
            int localeLen = fileName.lastIndexOf('.', fileName.length() );
            if( localeLen > 0){
                String locale = fileName.substring(0, localeLen);
                File affFile = new File(dictPath + '/' + locale + ".aff");
                if( affFile.exists() ){
                    try{
                        Hunspell.Dictionary dict = hunspell.getDictionary( dictPath + '/' + locale);
                        dictionaries.add(dict);
                        locales.add( locale );
                    }catch( UnsupportedEncodingException e){
                        LOG.error(e.getMessage());
                    }catch( FileNotFoundException e){
                        LOG.error(e.getMessage());
                  }
                }
            }
        }

    }
    /**
     * @param locale requested locale
     * @return null if locale is not supported, a dictionary otherwise
     */
    public Hunspell.Dictionary getDictionary( String locale ){
        Hunspell.Dictionary ret = null;
        if( locale.length() > 1 ){
            Iterator<Hunspell.Dictionary> it = dictionaries.iterator();
            while( it.hasNext() ){
                Hunspell.Dictionary dict = it.next();
                String lc = dict.getLocale().getCountry();
                String ls1 = locale.length() >= 5 ? locale.substring(3, 5) : new String();
                String ll = dict.getLocale().getLanguage();
                String ls2 = locale.substring(0, 2);
                if( lc.equals( ls1 ) &&
                    ll.equals( ls2 ) ){
                    ret = dict;
                    break;
                }
            }
        }
        return ret;
    }
    /**
     * @return list of dictionary locales
     */
    public List<String> getLocales(){
        return locales;
    }
}
